package org.accidia.echo;

import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.typesafe.config.ConfigFactory;
import org.accidia.echo.dao.StorageInitFailed;
import org.accidia.echo.protos.Protos.*;
import org.accidia.echo.services.IDataSourceService;
import org.accidia.echo.services.ITenantService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.*;

/**
 * Singleton 'context' objects, carrying configurations, injector, executor service pool, etc.
 * The context is the initializer of the application; it starts by loading configurations from
 * 'echo.conf' file, followed by loading Guice modules and initializing all tenants.
 */
public enum EchoContext {
    // the singleton instance
    INSTANCE;

    private static final Logger logger = LoggerFactory.getLogger(EchoContext.class);

    private ListeningExecutorService executorService;
    private EchoConfig configuration;
    private Injector injector;

    public void init() throws ReflectiveOperationException, StorageInitFailed {
        logger.info("initializing configurations...");
        initConfigurations();

        logger.info("initializing executor service...");
        initExecutorService();

        // create the Guice injector
        logger.info("initializing injector...");
        initInjector();

        // register default datasources given in the configurations
        logger.info("initializing default datasources...");
        initDefaultDataSources();

        // register default tenants given in the configurations
        logger.info("initializing default tenants...");
        initDefaultTenants();

        // register datasources stored in storage
        logger.info("initializing registered datasources...");
        initRegisteredDataSources();

        // register tenants stored in storage
        logger.info("initializing registered tenants...");
        initRegisteredTenants();
    }

    // reload the configuration
    public void reloadConfigurations(final String userConfigurationPath) {
        logger.debug("reloadConfigurations()");
        // TODO reload user configuration
//        this.userConfigurationPath = Strings.emptyToNull(userConfigurationPath);
    }

    /**
     * Access to the Guice injector object
     *
     * @return the Guice injector object
     */
    public Injector getInjector() {
        logger.debug("getInjector()");
        return this.injector;
    }

    /**
     * Access application wide configuration object
     *
     * @return configuration instance
     */
    public EchoConfig getConfiguration() {
        logger.debug("getConfiguration()");
        return this.configuration;
    }

    /**
     * Access application wide executor service
     *
     * @return the executor service
     */
    public ListeningExecutorService getExecutorService() {
        logger.debug("getExecutorService()");
        return this.executorService;
    }

    protected void initConfigurations() {
        this.configuration = EchoConfig.newInstanceForConfig(ConfigFactory.load(Constants.CONFIG__FILE_PATH));
    }

    protected void initExecutorService() {
        this.executorService = MoreExecutors.listeningDecorator(
                new ThreadPoolExecutor(0, 64,
                        60L, TimeUnit.SECONDS,
                        new LinkedBlockingDeque<>())
//                Executors.newWorkStealingPool()
        );
    }

    // read through module names given in the configuration and load them all
    @SuppressWarnings("unchecked")
    protected void initInjector() throws ReflectiveOperationException {
        checkState(this.configuration != null, "null configuration");

        final List<String> moduleNames = getConfiguration().getConfig().getStringList(Constants.CONFIG_KEY__MODULES);
        checkArgument(moduleNames != null && !moduleNames.isEmpty(), "null/empty storage modules");

        final List<Module> modules = new ArrayList<>(moduleNames.size() + 1);
        for (final String moduleName : moduleNames) {
            logger.info("adding module: {}", moduleName);
            modules.add(((Class<? extends Module>) Class.forName(moduleName)).newInstance());
        }

        logger.info("creating guice injector");
        this.injector = Guice.createInjector(modules);
    }

    protected void initDefaultTenants() throws StorageInitFailed {
        final List<Tenant> defaultTenants = getConfiguration().getDefaultTenants();
        doRegisterTenants(defaultTenants);
    }

    protected void initDefaultDataSources() throws StorageInitFailed {
        final Collection<DataSource> defaultDataSources = getInjector().getInstance(IDataSourceService.class)
                    .getDefaultDataSources();
        doRegisterDataSources(defaultDataSources);
    }

    protected void initRegisteredTenants() throws StorageInitFailed {
//        final List<Message> registeredTenants;
//        final List<Tenant> tenants;
//        try {
//            registeredTenants = getInjector().getInstance(ITenantService.class)
//                    .getObjectsServicesForTenant("tenant").getObjectList("todo", 0, -1).get();
//            tenants = new ArrayList<>(registeredTenants.size());
//            for (final Message message : registeredTenants) {
//                tenants.add(Tenant.newBuilder().mergeFrom(message.toByteArray()).buildPartial());
//            }
//        } catch (InterruptedException | ExecutionException | InvalidProtocolBufferException e) {
//            logger.error("failed to initialize registered tenants");
//            throw new StorageInitFailed(e);
//        }
//        doRegisterTenants(tenants);
    }

    protected void doRegisterTenants(final List<Tenant> tenants) {
        if (tenants == null || tenants.isEmpty()) {
            logger.info("no tenant to register -> ignoring");
            return;
        }

        final ITenantService tenantService = getInjector().getInstance(ITenantService.class);
        for (final Tenant tenant : tenants) {
            logger.info("registering tenant: {}", tenant.getName());
            try {
                tenantService.registerTenant(tenant);
            } catch (final Exception e) {
                logger.error("unable to register tenant -> rethrowing as runtime exception", e);
                throw new RuntimeException("cannot register tenant", e);
            }
        }
    }

    protected void initRegisteredDataSources() throws StorageInitFailed {
        final IDataSourceService dataSourceService = getInjector().getInstance(IDataSourceService.class);
        final Collection<DataSource> registeredDataSources = dataSourceService.getRegisteredDataSources();
        doRegisterDataSources(registeredDataSources);
    }

    protected void doRegisterDataSources(final Collection<DataSource> dataSources) {
        
    }
}

