package org.accidia.echo.mysql;

import com.mchange.v2.c3p0.ComboPooledDataSource;
import org.accidia.echo.protos.Protos.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.PropertyVetoException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * This class encapsulates all meta data required for initializing a JDBC connection
 * to MySQL.
 */
public class MySqlDataSource {

    private static final Logger logger = LoggerFactory.getLogger(MySqlDataSource.class);

    private static Map<DataSource, MySqlDataSource> instaces = new HashMap<>();

    private final ComboPooledDataSource connectoinPoolDataSource;
    private final DataSource datasource;

    public static MySqlDataSource newInstance(final DataSource dataSource) {
        return new MySqlDataSource(dataSource);
    }

    public synchronized static MySqlDataSource getInstance(final DataSource datasource) {
        if (!instaces.containsKey(datasource)) {
            instaces.put(datasource, newInstance(datasource));
        }
        return instaces.get(datasource);
    }

    protected MySqlDataSource(final DataSource datasource) {
        logger.debug("MySqlStorageMeta(storageMeta)");


        this.datasource = datasource;

        // data source
        this.connectoinPoolDataSource = new ComboPooledDataSource();
        try {
            this.connectoinPoolDataSource.setDriverClass("com.mysql.jdbc.Driver");
        } catch (final PropertyVetoException e) {
            throw new RuntimeException(e);
        }

        final String hostname = getMetadataValue(this.datasource.getMetadataList(), "hostname", true);
        final String port = getMetadataValue(this.datasource.getMetadataList(), "port", true);
        final String databaseName = getMetadataValue(this.datasource.getMetadataList(), "database", true);
        final String connectionProperties = getMetadataValue(this.datasource.getMetadataList(), "connection_properties", false);

        // jdbc url
        final String jdbcUrl = getJdbcUrl(hostname, port, databaseName, connectionProperties);
        logger.info("setting jdbc url to: {}", jdbcUrl);
        this.connectoinPoolDataSource.setJdbcUrl(jdbcUrl);

        // username
        final String username = getMetadataValue(this.datasource.getMetadataList(), "username", false);
        if (username != null) {
            logger.info("setting user to: {}", username);
            this.connectoinPoolDataSource.setUser(username);
        }

        // password
        final String password = getMetadataValue(this.datasource.getMetadataList(), "password", false);
        if (password != null) {
            logger.info("setting password...");
            this.connectoinPoolDataSource.setPassword(password);
        }

        // always test on checkout
        this.connectoinPoolDataSource.setTestConnectionOnCheckout(true);
    }

    public ComboPooledDataSource getConnectoinPoolDataSource() {
        logger.debug("getConnectoinPoolDataSource()");
        return this.connectoinPoolDataSource;
    }

    protected String getJdbcUrl(final String hostname, final String port,
                             final String databaseName, final String connectionProperties) {
        final String url = String.format(
                "jdbc:mysql://%s:%s/%s?%s", hostname, port, databaseName, connectionProperties
        );
        logger.info("jdbc url is: {}", url);
        return url;
    }

    protected String getMetadataValue(final List<DataSource.MetaData> metaData,
                                      final String metadataName,
                                      final boolean isRequired) {
        for (final DataSource.MetaData md : metaData) {
            if (md.getName().equalsIgnoreCase(metadataName)) {
                return md.getValue();
            }
        }
        checkArgument(isRequired, "invalid metadata: " + metadataName);
        return null;
    }
}

