package org.accidia.echo.mysql.keyvalue;

import com.google.common.base.Strings;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import org.accidia.echo.dao.IProtobufDao;
import org.accidia.echo.mysql.MySqlDataSource;
import org.accidia.echo.protos.Protos.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import java.io.IOException;
import java.util.*;

import static com.google.common.base.Preconditions.checkArgument;

public class MySqlKeyValueProtobufDao extends JdbcDaoSupport
        implements IProtobufDao {

    private static final Logger logger = LoggerFactory.getLogger(MySqlKeyValueProtobufDao.class);
    private final Message messageDefaultInstance;
    private final MySqlKeyValueProtobufRowMapper mySqlProtobufRowMapper = new MySqlKeyValueProtobufRowMapper();
    private final RowMapper<Message> rowMapper = (resultSet, rowNum)
            -> this.mySqlProtobufRowMapper.mapResultSet(resultSet, getMessageBuilder());

    public static MySqlKeyValueProtobufDao newInstance(final Message messageDefaultInstancee,
                                                       final DataSource dataSource) {
        // TODO validate
        try {
            return new MySqlKeyValueProtobufDao(messageDefaultInstancee, dataSource).createTableIfNotExist();
        } catch (Descriptors.DescriptorValidationException | ReflectiveOperationException | IOException e) {
            logger.warn("failed to create new mysql protobuf dao instance:", e);
            throw new RuntimeException(e);
        }
    }

    protected MySqlKeyValueProtobufDao(final Message messageDefaultInstance, final DataSource dataSource)
            throws Descriptors.DescriptorValidationException, ReflectiveOperationException, IOException {
        logger.debug("MySqlProtobufDao()");

        this.messageDefaultInstance = messageDefaultInstance;
        setDataSource(MySqlDataSource.getInstance(dataSource).getConnectoinPoolDataSource());
    }

    @Override
    public List<Message> getAll() {
        logger.debug("getAll()");
        return doGetAll();
    }

    @Override
    public Message findByKey(final String key) {
        logger.debug("findByKey(key)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        return doFindByKey(key);
    }

    @Override
    public Message findFieldsByKey(final String key, final List<String> fieldsIgnored) {
        logger.debug("findFieldsByKey()");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        return doFindByKey(key);
    }

    @Override
    public List<String> findList(final String listKey, final int start, final int count) {
        logger.debug("findList()");
        checkArgument(start >= 0, "invalid start");
        checkArgument(count >= -1, "invalid count");
        return doFindList(listKey, start, count);
    }

    @Override
    public void store(final String key, final Message object) {
        logger.debug("store()");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        checkArgument(object != null, "null object");

        doStore(key, object);
    }

    @Override
    public void addToList(final String listKey, final String objectKey) {
        checkArgument(!Strings.isNullOrEmpty(listKey), "null/empty listKey");
        checkArgument(!Strings.isNullOrEmpty(objectKey), "null/empty objectKey");
        doAddToList(listKey, objectKey);
    }

    @Override
    public void storeOrUpdate(final String key, final Message object) {
        // TODO this should be different from store
        store(key, object);
    }

    @Override
    public void archive(final String key) {
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        doArchive(key);
    }

    @Override
    public Message getMessageDefaultInstance() {
        return this.messageDefaultInstance;
    }

    protected List<Message> doGetAll() {
        final String sql = "SELECT `KEY`,`VALUE` FROM " + getTableName();
        final List<Message> messages = getJdbcTemplate().query(sql, this.rowMapper);
        if (messages == null) {
            return Collections.emptyList();
        }
        return messages;
    }

    protected void doArchive(final String key) {
        final String sql = "REPLACE INTO " + getArchiveTableName() + " SET `KEY` = ?";
        getJdbcTemplate().update(sql, key);
    }

    protected Message doFindByKey(final String key) {
        final String sql = "SELECT `KEY`,`VALUE` FROM " + getTableName() + " WHERE `KEY` = ?";
        final List<Message> messages = getJdbcTemplate().query(sql, this.rowMapper, key);
        if (messages == null || messages.isEmpty()) {
            return null;
        }
        return messages.get(0);
    }

    protected void doStore(final String key, final Message object) {
        final String sql = "REPLACE INTO " + getTableName() + " SET `KEY` = ?, `VALUE` = ?";
        getJdbcTemplate().update(sql, key, object.toByteArray());
    }

    protected List<String> doFindList(final String listKey, final int start, final int count) {
        final StringBuilder sqlStringBuilder = new StringBuilder();
        final List<Object> parameters = new ArrayList<>();
        parameters.add(listKey);
        sqlStringBuilder.append("SELECT OBJECT_KEY FROM ")
                .append(getListTableName())
                .append(" WHERE LIST_KEY = ? ");
        if (start > 0) {
            sqlStringBuilder.append("LIMIT ?");
            parameters.add(start);
        }
        if (count > 0) {
            sqlStringBuilder.append(",?");
            parameters.add(count);
        }
        final String sql = sqlStringBuilder.toString();
        logger.info("sql to run is {} and list key is {}", sql, listKey);

        final List<String> objectKeys = getJdbcTemplate().query(sql, (rs, rowNum) -> rs.getString("OBJECT_KEY"), parameters.toArray());
        logger.info("object keys: {}", objectKeys);
        return objectKeys != null ? objectKeys : Collections.emptyList();
    }

    protected void doAddToList(final String listKey, final String objectKey) {
        final StringBuilder sqlStringBuilder = new StringBuilder();
        sqlStringBuilder.append("REPLACE INTO ")
                .append(getListTableName())
                .append(" SET `LIST_KEY` = ?, `OBJECT_KEY` = ?");
        getJdbcTemplate().update(sqlStringBuilder.toString(), listKey, objectKey);
    }

    protected MySqlKeyValueProtobufDao createTableIfNotExist() {
        final String sql = "CREATE TABLE IF NOT EXISTS `" + getTableName() +
                "` ( `KEY` varchar(128) NOT NULL, `VALUE` blob NOT NULL, PRIMARY KEY (`KEY`) ) " +
                " ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        final String archiveSql = "CREATE TABLE IF NOT EXISTS `" + getArchiveTableName() +
                "` ( `KEY` varchar(128) NOT NULL, `TIMESTAMP` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP," +
                "  PRIMARY KEY (`KEY`)" +
                ") ENGINE=InnoDB DEFAULT CHARSET=utf8";

        final String listSql = "CREATE TABLE IF NOT EXISTS `" + getListTableName() +
                "` ( `LIST_KEY` varchar(128) NOT NULL, `OBJECT_KEY` varchar(128) NOT NULL, " +
                " PRIMARY KEY (`LIST_KEY`,`OBJECT_KEY`) ) " +
                " ENGINE=InnoDB DEFAULT CHARSET=utf8";

        getJdbcTemplate().update(sql);
        getJdbcTemplate().update(archiveSql);
        getJdbcTemplate().update(listSql);
        return this;
    }

    protected Message.Builder getMessageBuilder() {
        return this.messageDefaultInstance.newBuilderForType();
    }

    // by conventions, table names must be the same as the message class name
    protected String getTableName() {
        return getMessageDefaultInstance().getClass().getSimpleName().toUpperCase();
    }

    // by conventions, table names must be the same as the message class name
    protected String getListTableName() {
        return getTableName() + "_LIST";
    }
    
    protected String getArchiveTableName() {
        return getTableName() + "_ARCHIVE";
    }
}

