package org.accidia.echo.dao;

import com.google.inject.AbstractModule;
import org.accidia.echo.EchoContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.accidia.echo.protos.Protos;

import java.util.List;

import static com.google.common.base.Preconditions.*;

public abstract class AbstractStorageModule extends AbstractModule {
    private final Logger logger = LoggerFactory.getLogger(AbstractStorageModule.class);

    public abstract void configureForDataSource(final Protos.DataSource dataSource) throws StorageInitFailed;

    /**
     * Storage modules do not implement 'configure'; instead 'configureForDataSource' is used.
     */
    @Override
    protected final void configure() {
        logger.debug("configure()");

        checkArgument(EchoContext.INSTANCE.getConfiguration() != null, "null configuration");
        final List<Protos.DataSource> defaultDataSources = EchoContext.INSTANCE.getConfiguration().getDefaultDataSources();
        checkArgument(defaultDataSources != null && !defaultDataSources.isEmpty(),
                "null/empty list of storage meta");

        for (final Protos.DataSource dataSource : defaultDataSources) {
            logger.info("trying to configure for store with name {}", dataSource.getName());
            try {
                // check if this is the module to load the store meta
                if (getStorageType() == dataSource.getStorageType()) {
                    logger.info("configuring store with name {}", dataSource.getName());
                    // yes
                    configureForDataSource(dataSource);
                } else {
                    logger.info("module {} is not responsible for the store with name {}",
                            getClass().getName(), dataSource.getName());
                }
            } catch (final StorageInitFailed e) {
                logger.error("configuring storage failed: ", e);
                throw new RuntimeException(e);
            }
        }
    }

    protected abstract Protos.DataSource.StorageType getStorageType();
}

