package org.accidia.echo;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import org.accidia.echo.protos.Protos.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.*;

public class EchoConfig {
    private static final Logger logger = LoggerFactory.getLogger(EchoConfig.class);
    private final Config config;

    private final String bind;
    private final int timeoutInSeconds;

    private EchoConfig(final Config config) {
        logger.debug("EchoConfig()");
        this.config = config;

        // init bind
        this.bind = checkNotNull(getConfig().getString(Constants.CONFIG_KEY__BIND),
                "missing config: " + Constants.CONFIG_KEY__BIND);

        // init timeout in seconds
        this.timeoutInSeconds = checkNotNull(getConfig().getInt(Constants.CONFIG_KEY__RESOURCE_TIMEOUT_SECONDS),
                "missing config: " + Constants.CONFIG_KEY__RESOURCE_TIMEOUT_SECONDS);
    }

    public static EchoConfig newInstanceForConfig(final Config config) {
        logger.debug("newInstanceForConfig(config)");
        checkArgument(config != null, "null config");
        return new EchoConfig(config);
    }

    public String getBind() {
        logger.debug("getBind()");
        return this.bind;
    }

    public int getTimeoutInSeconds() {
        logger.debug("getTimeoutInSeconds()");
        return this.timeoutInSeconds;
    }

    public List<Tenant> getDefaultTenants() {
        logger.debug("getDefaultTenants()");

        final List<Tenant> protobufTypeMetaList = new ArrayList<>();
        for (final ConfigObject config : getConfig().getObjectList(Constants.CONFIG_KEY__TENANTS)) {
            final String tenantName = config.toConfig().getString(Constants.CONFIG_KEY__TENANT__NAME);
            final String dataSourceName = config.toConfig().getString(Constants.CONFIG_KEY__TENANT__DATASOURCE_NAME);
            final String namespace = config.toConfig().getString(Constants.CONFIG_KEY__TENANT__NAMESPACE);
            final String protobufMessage = config.toConfig().getString(Constants.CONFIG_KEY__TENANT__PROTOBUF_MESSAGE);

            final Tenant tenant = Tenant.newBuilder()
                    .setName(tenantName)
                    .setDatasourceName(dataSourceName)
                    .setNamespace(namespace)
                    .setProtobufMessage(protobufMessage)
                    .build();
            logger.info("adding tenant: {}", tenant);
            protobufTypeMetaList.add(tenant);
        }
        return protobufTypeMetaList;
    }

    public List<DataSource> getDefaultDataSources() {
        logger.debug("getDefaultDataSources()");

        final List<DataSource> defaultDataSources = new ArrayList<>();
        for (final ConfigObject config : getConfig().getObjectList(Constants.CONFIG_KEY__DATA_SOURCES)) {
            final String dataSourceName = config.toConfig().getString(Constants.CONFIG_KEY__DATA_SOURCE__NAME);
            final String storageType = config.toConfig().getString(Constants.CONFIG_KEY__DATA_SOURCE__STORAGE_TYPE);
            final boolean isReadOnly = config.toConfig().getBoolean(Constants.CONFIG_KEY__DATA_SOURCE__IS_READONLY);

            final DataSource.Builder dataSourceBuilder = DataSource.newBuilder()
                    .setName(dataSourceName)
                    .setStorageType(DataSource.StorageType.valueOf(storageType.toUpperCase()))
                    .setReadonly(isReadOnly);
            final ConfigObject metaConfigObject = config.toConfig().getObject(Constants.CONFIG_KEY__DATA_SOURCE__META);
            for (final Map.Entry<String, ConfigValue> entry : metaConfigObject.entrySet()) {
                dataSourceBuilder.addMetadata(DataSource.MetaData.newBuilder()
                        .setName(entry.getKey())
                        .setValue(String.valueOf(entry.getValue().unwrapped()))
                        .build());
            }
            final DataSource dataSource = dataSourceBuilder.build();
            logger.info("adding datasource: {}", dataSource);
            defaultDataSources.add(dataSource);
        }
        return defaultDataSources;
    }

    public Config getConfig() {
        logger.debug("getConfig()");
        return this.config;
    }
}

