package org.accidia.echo.memcache;

import com.google.common.base.Preconditions;
import net.spy.memcached.MemcachedClient;
import org.accidia.echo.protos.Protos.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.PropertyVetoException;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class MemcacheDataSource {
    private static final Logger logger = LoggerFactory.getLogger(MemcacheDataSource.class);
    private static Map<DataSource, MemcacheDataSource> instaces = new HashMap<>();
    private final DataSource datasource;
    private final MemcachedClient memcachedClient;

    public synchronized static MemcacheDataSource getInstance(final DataSource datasource) throws IOException {
        if (!instaces.containsKey(datasource)) {
            instaces.put(datasource, newInstance(datasource));
        }
        return instaces.get(datasource);
    }

    public static MemcacheDataSource newInstance(final DataSource dataSource) throws IOException {
        return new MemcacheDataSource(dataSource);
    }

    protected MemcacheDataSource(final DataSource datasource) throws IOException {
        logger.debug("MemcacheDataSource(storageMeta)");
        this.datasource = datasource;

        // data source
        final String hostname = getMetadataValue(this.datasource.getMetadataList(), "hostname", true);
        final int port = Integer.parseInt(getMetadataValue(this.datasource.getMetadataList(), "port", true));
        this.memcachedClient = new MemcachedClient(new InetSocketAddress(hostname, port));
    }

    protected String getMetadataValue(final List<DataSource.MetaData> metaData,
                                      final String metadataName,
                                      final boolean isRequired) {
        for (final DataSource.MetaData md : metaData) {
            if (md.getName().equalsIgnoreCase(metadataName)) {
                return md.getValue();
            }
        }
        Preconditions.checkArgument(isRequired, "invalid metadata: " + metadataName);
        return null;
    }

    public MemcachedClient getMemcachedClient() {
        return memcachedClient;
    }
}
