package org.accidia.echo.dao;

import com.google.protobuf.Message;

import java.util.List;

public interface IProtobufDao {

    /**
     * Returns the object assigned to the given key
     *
     * @param key key assigned to the object to be fetched
     * @return the protobuf object assigned to the given key if found, null otherwise
     */
    Message findByKey(final String key);

    /**
     * Returns partial object containing only the specified fields
     *
     * @param key    key assigned to the object to be fetched
     * @param fields list of field names to be returned
     * @return partial message object
     */
    Message findFieldsByKey(final String key, final List<String> fields);

    /**
     * Returns list of objects starting from the given index; the list
     * will be maximum of size 'count'
     *
     * @param listKey the list key
     * @param start   the starting index
     * @param count   the maximum count of objects to return
     * @return list of object keys
     */
    List<String> findList(final String listKey, final int start, final int count);

    List<String> findAllList(final String listKey);


    List<Message> findListObjects(final String listKey, int start, int count);

    List<Message> findAllListObjects(final String listKey);


    List<Message> findOrderedListObjectsAscending(final String listKey, int start, int count);

    List<Message> findAllOrderedListObjectsAscending(final String listKey);


    List<Message> findOrderedListObjectsDescending(final String listKey, int start, int count);

    List<Message> findAllOrderedListObjectsDescending(final String listKey);


    List<String> findOrderedListAscending(final String listKey, final int start, final int conut);

    List<String> findAllOrderedListAscending(final String listKey);


    List<String> findOrderedListDescending(final String listKey, final int start, final int conut);

    List<String> findAllOrderedListDescending(final String listKey);

    /**
     * Store the given object
     *
     * @param key    the key assigned to the object
     * @param object the protobuf object to be stored
     */
    void store(final String key, final Message object);

    /**
     * Delete the given object
     *
     * @param key    the key assigned to the object
     */
    void delete(final String key);

    /**
     * Add an object to a list
     *
     * @param listKey   key associated to the list
     * @param objectKey object key
     */
    void addToList(final String listKey, final String objectKey);

    /**
     * Remove an object from a list
     *
     * @param listKey   key associated to the list
     * @param objectKey object key
     */
    void removeFromList(final String listKey, final String objectKey);

    void addToOrderedList(final String listKey, final String objectKey, final long weight);

    /**
     * Store or update the given object
     *
     * @param key    the key assigned to the object
     * @param object the protobuf object to be stored
     */
    void storeOrUpdate(final String key, final Message object);

    Message getMessageDefaultInstance();
}

