package org.accidia.echo.mysql.relational;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import org.accidia.echo.dao.IProtobufDao;
import org.accidia.echo.mysql.EchoSqls;
import org.accidia.echo.mysql.MySqlDataSource;
import org.accidia.echo.protobuf.Protobufs;
import org.accidia.echo.protos.Protos.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;

public class MySqlProtobufDao extends JdbcDaoSupport implements IProtobufDao {

    private static final Logger logger = LoggerFactory.getLogger(MySqlProtobufDao.class);
    private final Message messageDefaultInstance;
    private final MySqlProtobufRowMapper mySqlProtobufRowMapper = new MySqlProtobufRowMapper();

    private final RowMapper<Message> rowMapper = (resultSet, rowNum)
            -> this.mySqlProtobufRowMapper.mapResultSet(resultSet, getMessageBuilder());

    public static MySqlProtobufDao newInstance(final Message messageDefaultInstance, final DataSource dataSource) {
        // TODO validate
        try {
            return new MySqlProtobufDao(messageDefaultInstance, dataSource);
        } catch (Descriptors.DescriptorValidationException | ReflectiveOperationException | IOException e) {
            logger.warn("failed to create new mysql protobuf dao instance:", e);
            throw new RuntimeException(e);
        }
    }

    protected MySqlProtobufDao(final Message messageDefaultInstance, final DataSource dataSource)
            throws Descriptors.DescriptorValidationException, ReflectiveOperationException, IOException {
        logger.debug("MySqlProtobufDao()");

        final MySqlDataSource mySqlDataSource = MySqlDataSource.newInstance(dataSource);
        setDataSource(mySqlDataSource.getConnectoinPoolDataSource());
        this.messageDefaultInstance = messageDefaultInstance;
    }

    @Override
    public Message findByKey(final String key) {
        logger.debug("findByKey(key)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        return doFindByKey(key);
    }

    protected Message doFindByKey(final String key) {
        final List<String> fieldNames = Protobufs.getAllFieldNamesToUpperCase(getMessageDefaultInstance());
        final StringBuilder sqlStringBuilder = new StringBuilder();
        sqlStringBuilder.append("SELECT ")
                .append(Joiner.on(",").join(fieldNames))
                .append(" FROM ")
                .append(getTableName())  // namespace is used as table name for mysql
                .append(" WHERE ")
                .append(getKeyFieldName())
                .append(" = ?");

        final List<Message> messages = getJdbcTemplate().query(sqlStringBuilder.toString(), this.rowMapper, key);
        if (messages == null || messages.isEmpty()) {
            return null;
        }
        return messages.get(0);
    }

    @Override
    public Message findFieldsByKey(final String key, final List<String> fields) {
        logger.debug("findFieldsByKey(key,fields)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        return doFindFieldsByKey(key, fields);
    }

    protected Message doFindFieldsByKey(final String key, final List<String> fields) {
        final String sql = EchoSqls.getSqlSelectFieldsForMessage(getMessageDefaultInstance(), fields)
                + " FROM " +  getTableName()
                + " WHERE " + getKeyFieldName() + " = ?";

        final List<Message> messages = getJdbcTemplate().query(sql, this.rowMapper, key);
        if (messages == null || messages.isEmpty()) {
            return null;
        }
        return messages.get(0);
    }

    @Override
    public List<String> findList(final String listKey, final int start, final int count) {
        logger.debug("findList(start,count,orderby)");
        checkArgument(start >= 0, "invalid start");
        checkArgument(count >= -1, "invalid count");
        return doFindList(listKey, start, count);
    }

    @Override
    public List<String> findAllList(String listKey) {
        return null;
    }

    @Override
    public List<Message> findListObjects(String listKey, int start, int count) {
        // TODO
        return null;
    }

    @Override
    public List<Message> findAllListObjects(String listKey) {
        return null;
    }

    @Override
    public List<Message> findOrderedListObjectsAscending(String listKey, int start, int count) {
        return null;
    }

    @Override
    public List<Message> findAllOrderedListObjectsAscending(String listKey) {
        return null;
    }

    @Override
    public List<Message> findOrderedListObjectsDescending(String listKey, int start, int count) {
        return null;
    }

    @Override
    public List<Message> findAllOrderedListObjectsDescending(String listKey) {
        return null;
    }

    @Override
    public List<String> findOrderedListAscending(String listKey, int start, int conut) {
        return null;
    }

    @Override
    public List<String> findAllOrderedListAscending(String listKey) {
        return null;
    }

    @Override
    public List<String> findOrderedListDescending(String listKey, int start, int conut) {
        return null;
    }

    @Override
    public List<String> findAllOrderedListDescending(String listKey) {
        return null;
    }

    @Override
    public List<String> findAllListsForObject(String objectKey) {
        return null;
    }

    @Override
    public void store(final String key, final Message object) {
        logger.debug("store(key,object)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        checkArgument(object != null, "null object");
        doStore(key, object);
    }

    @Override
    public void delete(String key) {

    }

    @Override
    public void addToList(final String listKey, final String objectKey) {
        logger.debug("addToList(listKey,objectKey)");
        checkArgument(!Strings.isNullOrEmpty(listKey), "null/empty listKey");
        checkArgument(!Strings.isNullOrEmpty(objectKey), "null/empty objectKey");
        doAddToList(listKey, objectKey);
    }

    @Override
    public void removeFromList(final String listKey, final String objectKey) {
        // TODO
    }

    @Override
    public void removeFromAllLists(String objectKey) {
        // TODO
    }

    @Override
    public void addToOrderedList(final String listKey, final String objectKey, long weight) {
        // TODO
    }

    @Override
    public long getObjectWeight(String listKey, String objectKey) {
        // TODO
        return 0;
    }

    @Override
    public void addToObjectWeight(String listKey, String objectKey, long count) {
        // TODO
    }

    @Override
    public void storeOrUpdate(final String key, final Message object) {
        // TODO this should be different from store
        store(key, object);
    }

    protected void doStore(final String key, final Message object) {
        logger.debug("storing message: {}", object);
        final StringBuilder sqlStringBuilder = new StringBuilder();
        sqlStringBuilder.append("REPLACE INTO ")
                .append(getTableName())
                .append(" SET ")
                .append(getKeyFieldName())
                .append(" = ?");

        for (final String fieldName : Protobufs.getDefinedFieldNames(object)) {
            if (fieldName.equalsIgnoreCase(getKeyFieldName())) {
                continue;
            }
            sqlStringBuilder.append(", ").append(fieldName.toUpperCase()).append(" = ?");
        }
        final String sql = sqlStringBuilder.toString();
        logger.debug("sql to run for store: {}", sql);

        final List<Object> parameters = new ArrayList<>();
        parameters.add(key);
        for (final String fieldName : Protobufs.getDefinedFieldNames(object)) {
            if (fieldName.equalsIgnoreCase(getKeyFieldName())) {
                continue;
            }
            parameters.add(Protobufs.getValueForFieldName(object, fieldName));
        }
        getJdbcTemplate().update(sqlStringBuilder.toString(), parameters.toArray());
    }

    protected List<String> doFindList(final String listKey, final int start, final int count) {
        final StringBuilder sqlStringBuilder = new StringBuilder();
        sqlStringBuilder.append("SELECT `OBJECT_KEY` FROM ")
                .append(getListTableName())
                .append(" WHERE `LIST_KEY` = ? LIMIT ?,?");

        final List<String> objectKeys = getJdbcTemplate()
                .query(sqlStringBuilder.toString(), (rs, rowNum) -> rs.getString(0), listKey, start, count);
        return objectKeys != null ? objectKeys : Collections.emptyList();
    }

    protected void doAddToList(final String listKey, final String objectKey) {
        final StringBuilder sqlStringBuilder = new StringBuilder();
        sqlStringBuilder.append("REPLACE INTO ")
                .append(getListTableName())
                .append(" SET `LIST_KEY` = ?, `OBJECT_KEY` = ?");
        getJdbcTemplate().update(sqlStringBuilder.toString(), listKey, objectKey);
    }

    @Override
    public Message getMessageDefaultInstance() {
        return this.messageDefaultInstance;
    }

    protected Message.Builder getMessageBuilder() {
        return this.messageDefaultInstance.newBuilderForType();
    }

    // by convention, the field with index 0 is always the key
    protected String getKeyFieldName() {
        return getMessageDefaultInstance().getDescriptorForType().getFields().get(0).getName().toUpperCase();
    }

    // by conventions, table names must be the same as the message class name
    protected String getTableName() {
        return getMessageDefaultInstance().getClass().getSimpleName().toUpperCase();
    }

    // by conventions, table names must be the same as the message class name
    protected String getListTableName() {
        return getTableName() + "_LIST";
    }
}

