package org.accidia.echo.mysql.keyvalue;

import com.google.common.base.Preconditions;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import org.accidia.echo.dao.IProtobufDao;
import org.accidia.echo.memcache.MemcacheAccessor;
import org.accidia.echo.memcache.MemcacheDataSource;
import org.accidia.echo.protos.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;

public class MemcacheOnMySqlProtobufDao<T extends MySqlKeyValueProtobufDao> implements IProtobufDao {
    private static final Logger logger = LoggerFactory.getLogger(MemcacheOnMySqlProtobufDao.class);
    private final T mySqlKeyValueProtobufDao;
    private final MemcacheAccessor memcacheAccessor;
    private final int cacheTimeoutMs;

    public static <T extends MySqlKeyValueProtobufDao> MemcacheOnMySqlProtobufDao<T> newInstance(final Protos.DataSource memcacheDataSource,
                                                                                                 final T mySqlKeyValueProtobufDao)
            throws IOException, ReflectiveOperationException, Descriptors.DescriptorValidationException {
        return newInstance(memcacheDataSource, mySqlKeyValueProtobufDao, 0);
    }

    public static <T extends MySqlKeyValueProtobufDao> MemcacheOnMySqlProtobufDao<T> newInstance(final Protos.DataSource memcacheDataSource,
                                                                                                 final T mySqlKeyValueProtobufDao,
                                                                                                 final int cacheTimeoutMs)
            throws IOException, ReflectiveOperationException, Descriptors.DescriptorValidationException {
        Preconditions.checkArgument(cacheTimeoutMs >= 0, "invalid cache timeout");
        return new MemcacheOnMySqlProtobufDao<>(memcacheDataSource, mySqlKeyValueProtobufDao, cacheTimeoutMs);
    }

    protected MemcacheOnMySqlProtobufDao(final Protos.DataSource memcacheDataSource, final T mySqlKeyValueProtobufDao, final int cacheTimeoutMs)
            throws Descriptors.DescriptorValidationException, ReflectiveOperationException, IOException {
        this.mySqlKeyValueProtobufDao = mySqlKeyValueProtobufDao;
        this.memcacheAccessor = MemcacheAccessor.getInstance(MemcacheDataSource.getInstance(memcacheDataSource));
        this.cacheTimeoutMs = cacheTimeoutMs;
    }

    @Override
    public Message findByKey(final String key) {
        Message message = null;
        try {
            message = this.memcacheAccessor.get(key, 50); // timeout on 50 milliseconds
        } catch (RuntimeException e) {
            logger.warn("exception on memcache get for key: " + key, e);
        }
        if (message != null) {
            logger.info("cache hit on key: {}", key);
            return message;
        }
        logger.info("cache miss on key: {}", key);

        message = this.mySqlKeyValueProtobufDao.findByKey(key);
        if (message != null) {
            this.memcacheAccessor.set(key, this.cacheTimeoutMs, message);
        }
        return message;
    }

    @Override
    public Message findFieldsByKey(final String key, final List<String> fields) {
        return this.mySqlKeyValueProtobufDao.findFieldsByKey(key, fields);
    }

    @Override
    public List<String> findList(final String listKey, int start, int count) {
        return this.mySqlKeyValueProtobufDao.findList(listKey, start, count);
    }

    @Override
    public List<String> findAllList(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllList(listKey);
    }

    @Override
    public List<Message> findListObjects(final String listKey, int start, int count) {
        return this.mySqlKeyValueProtobufDao.findListObjects(listKey, start, count);
    }

    @Override
    public List<Message> findAllListObjects(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllListObjects(listKey);
    }

    @Override
    public List<Message> findOrderedListObjectsAscending(final String listKey, int start, int count) {
        return this.mySqlKeyValueProtobufDao.findOrderedListObjectsAscending(listKey, start, count);
    }

    @Override
    public List<Message> findAllOrderedListObjectsAscending(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllOrderedListObjectsAscending(listKey);
    }

    @Override
    public List<Message> findOrderedListObjectsDescending(final String listKey, int start, int count) {
        return this.mySqlKeyValueProtobufDao.findOrderedListObjectsDescending(listKey, start, count);
    }

    @Override
    public List<Message> findAllOrderedListObjectsDescending(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllOrderedListObjectsDescending(listKey);
    }

    @Override
    public List<String> findOrderedListAscending(final String listKey, int start, int count) {
        return this.mySqlKeyValueProtobufDao.findOrderedListAscending(listKey, start, count);
    }

    @Override
    public List<String> findAllOrderedListAscending(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllOrderedListAscending(listKey);
    }

    @Override
    public List<String> findOrderedListDescending(final String listKey, int start, int conut) {
        return this.mySqlKeyValueProtobufDao.findOrderedListDescending(listKey, start, conut);
    }

    @Override
    public List<String> findAllOrderedListDescending(final String listKey) {
        return this.mySqlKeyValueProtobufDao.findAllOrderedListDescending(listKey);
    }

    @Override
    public List<String> findAllListsForObject(final String objectKey) {
        return this.mySqlKeyValueProtobufDao.findAllListsForObject(objectKey);
    }

    @Override
    public void store(final String key, final Message object) {
        this.mySqlKeyValueProtobufDao.store(key, object);
        this.memcacheAccessor.delete(key);
    }

    @Override
    public void delete(final String key) {
        this.memcacheAccessor.delete(key);
        this.mySqlKeyValueProtobufDao.delete(key);
    }

    @Override
    public void addToList(final String listKey, final String objectKey) {
        this.mySqlKeyValueProtobufDao.addToList(listKey, objectKey);
    }

    @Override
    public void removeFromList(final String listKey, final String objectKey) {
        this.mySqlKeyValueProtobufDao.removeFromList(listKey, objectKey);
    }

    @Override
    public void removeFromAllLists(final String objectKey) {
        this.mySqlKeyValueProtobufDao.removeFromAllLists(objectKey);
    }

    @Override
    public void addToOrderedList(final String listKey, final String objectKey, long weight) {
        this.mySqlKeyValueProtobufDao.addToOrderedList(listKey, objectKey, weight);
    }

    @Override
    public void storeOrUpdate(final String key, final Message object) {
        this.mySqlKeyValueProtobufDao.storeOrUpdate(key, object);
        this.memcacheAccessor.delete(key);
    }

    @Override
    public Message getMessageDefaultInstance() {
        return this.mySqlKeyValueProtobufDao.getMessageDefaultInstance();
    }
}

