package org.accidia.protoserver;

import org.accidia.protoserver.misc.ProtoServerException;
import org.eclipse.jetty.server.Server;
import org.glassfish.jersey.jetty.JettyHttpContainerFactory;
import org.glassfish.jersey.server.ResourceConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkState;

/**
 * <p>Abstract AbstractBaseProtoServerApplication class.</p>
 *
 * @author peymant
 * @version $Id: $Id
 */
public abstract class AbstractBaseProtoServerApplication implements IProtoServerApplication {
    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final ResourceConfig resourceConfig = getResourceConfig()
            .packages("org.accidia.protoserver");

    private Server server = JettyHttpContainerFactory.createServer(
            getBaseUri(), this.resourceConfig,
            false  // do not start
    );

    /** {@inheritDoc} */
    @Override
    public void startServer() throws ProtoServerException {
        checkState(this.server != null, "null server");
        doStartServer();
    }

    /** {@inheritDoc} */
    @Override
    public void joinOnServer() throws ProtoServerException {
        checkState(this.server != null, "null server");
        doJoinOnServer();
    }

    /** {@inheritDoc} */
    @Override
    public void stopServer() throws ProtoServerException {
        checkState(this.server != null, "null server");
        doStopServer();
    }

    /** {@inheritDoc} */
    @Override
    public Server getServer() {
        return this.server;
    }

    /**
     * <p>doStartServer.</p>
     *
     * @throws org.accidia.protoserver.misc.ProtoServerException if any.
     */
    protected void doStartServer() throws ProtoServerException {
        logger.info("starting protoserver server at {}", getBaseUri().toString());
        try {
            this.server.start();
        } catch (Exception e) {
            logger.error("exception caught at server.start -> rethrowing", e);
            throw new ProtoServerException(e);
        }
    }

    /**
     * <p>doJoinOnServer.</p>
     *
     * @throws org.accidia.protoserver.misc.ProtoServerException if any.
     */
    protected void doJoinOnServer() throws ProtoServerException {
        logger.info("joining on server");
        try {
            this.server.join();
        } catch (Exception e) {
            logger.error("exception caught at server.join -> rethrowing", e);
            throw new ProtoServerException(e);
        }
    }

    /**
     * <p>doStopServer.</p>
     *
     * @throws org.accidia.protoserver.misc.ProtoServerException if any.
     */
    protected void doStopServer() throws ProtoServerException {
        logger.info("stopping server");
        try {
            this.server.getServer().stop();
            this.server.destroy();
        } catch (Exception e) {
            logger.error("exception caught at server.stop/destroy -> rethrowing", e);
            throw new ProtoServerException(e);
        }
    }
}

