package org.adridadou.ethereum.ethereumj.config;

import java.util.Set;

/**
 * Created by davidroon on 18.09.16.
 * This code is released under Apache 2 license
 *
 * peer {
 * # The protocols supported by peer
 * # can be: [eth, shh, bzz]
 * capabilities = [eth]
 *
 * # Port on which ethereumj will listen
 * # for incoming connections
 * listen.port = 30303
 *
 * # connection timeout for trying to
 * # connect to a peer [seconds]
 * connection.timeout = 2
 *
 * # how much time [seconds]
 * # we will wait for a message
 * # to arrive before closing the channel
 * channel.read.timeout = 90
 *
 * # Private key of the peer
 * # The key is generated by default on the first run and stored in the database folder
 * # If you have your own peer ID, specify its private key here
 * # derived nodeId = dead745c1dbcde518b48e52aca1e8d5ba666005a2c8804e39826c6080fb11c1e8abe41d1e41896e871f204f790a90fa9781744cccecf492212192a7c56e7673b
 * # privateKey = f67c4032a7ff79bbfa7a780331b235c4eb681d51a0704cb1562064fb6c4bced4
 *
 * # Network id
 * networkId = 1
 *
 * p2p {
 * # the default version outbound connections are made with
 * # inbound connections are made with the version declared by the remote peer (if supported)
 * # version = 4
 *
 * # max frame size in bytes when framing is enabled
 * framing.maxSize = 32768
 *
 * # forces peer to send Handshake message in format defined by EIP-8,
 * # see https://github.com/ethereum/EIPs/blob/master/EIPS/eip-8.md
 * eip8 = true
 * }
 *
 * # max number of active peers our node will maintain
 * # extra peers trying to connect us will be dropped with TOO_MANY_PEERS message
 * # the incoming connection from the peer matching 'peer.trusted' entry is always accepted
 * maxActivePeers = 30
 * }
 */
public class PeerConfig {
    private final Boolean eip8;
    private final Integer framingMaxSize;
    private final Integer p2pVersion;
    private final Integer networkId;
    private final String privateKey;
    private final Integer channelReadTimeout;
    private final Integer connectionTimeout;
    private final Integer port;
    private final Set<EthereumProtocol> capabilities;

    public PeerConfig(Boolean eip8, Integer framingMaxSize, Integer p2pVersion, Integer networkId, String privateKey, Integer channelReadTimeout, Integer connectionTimeout, Integer port, Set<EthereumProtocol> capabilities) {
        this.eip8 = eip8;
        this.framingMaxSize = framingMaxSize;
        this.p2pVersion = p2pVersion;
        this.networkId = networkId;
        this.privateKey = privateKey;
        this.channelReadTimeout = channelReadTimeout;
        this.connectionTimeout = connectionTimeout;
        this.port = port;
        this.capabilities = capabilities;
    }

    @Override
    public String toString() {
        return "{" +
                "eip8=" + eip8 +
                "framingMaxSize=" + framingMaxSize +
                "p2pVersion=" + p2pVersion +
                "networkId=" + networkId +
                "privateKey='" + privateKey + "'" +
                "channelReadTimeout=" + channelReadTimeout +
                "connectionTimeout=" + connectionTimeout +
                "port=" + port +
                "capabilities=" + capabilities +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Boolean eip8;
        private Integer framingMaxSize;
        private Integer p2pVersion;
        private Integer networkId;
        private String privateKey;
        private Integer channelReadTimeout;
        private Integer connectionTimeout;
        private Integer port;
        private Set<EthereumProtocol> capabilities;

        public Builder eip8(Boolean eip8) {
            this.eip8 = eip8;
            return this;
        }

        public Builder framingMaxSize(Integer framingMaxSize) {
            this.framingMaxSize = framingMaxSize;
            return this;
        }

        public Builder p2pVersion(Integer p2pVersion) {
            this.p2pVersion = p2pVersion;
            return this;
        }

        public Builder networkId(Integer networkId) {
            this.networkId = networkId;
            return this;
        }

        public Builder privateKey(String privateKey) {
            this.privateKey = privateKey;
            return this;
        }

        public Builder channelReadTimeout(Integer channelReadTimeout) {
            this.channelReadTimeout = channelReadTimeout;
            return this;
        }

        public Builder connectionTimeout(Integer connectionTimeout) {
            this.connectionTimeout = connectionTimeout;
            return this;
        }

        public Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public Builder capabilities(Set<EthereumProtocol> capabilities) {
            this.capabilities = capabilities;
            return this;
        }

        public PeerConfig build() {
            return new PeerConfig(eip8, framingMaxSize, p2pVersion, networkId, privateKey, channelReadTimeout, connectionTimeout, port, capabilities);
        }
    }

}
