package org.miniclient.json.parser;

import static org.miniclient.json.common.TokenPool.TOKEN_EOF;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.miniclient.json.LiteJsonTokenizer;
import org.miniclient.json.MiniClientJsonException;
import org.miniclient.json.common.CharQueue;
import org.miniclient.json.common.CharacterUtil;
import org.miniclient.json.common.CyclicCharArray;
import org.miniclient.json.common.JsonToken;
import org.miniclient.json.common.LiteralUtil;
import org.miniclient.json.common.Literals;
import org.miniclient.json.common.Symbols;
import org.miniclient.json.common.TokenPool;
import org.miniclient.json.common.TokenTypes;
import org.miniclient.json.common.UnicodeUtil;


// Base class for JsonTokenizer implementation.
// TBD: 
// Make a tokenzier reusable (without having to create a new tokenizer for every parse()'ing.) ???
// ...
// Note:
// Current implementation is not thread-safe (and, probably, never will be).
// 
public final class MiniClientJsonTokenizer implements LiteJsonTokenizer
{
    private static final Logger log = Logger.getLogger(MiniClientJsonTokenizer.class.getName());

    // MAX_STRING_LOOKAHEAD_SIZE should be greater than 6.
    private static final int MAX_STRING_LOOKAHEAD_SIZE = 128;   // temporary
    // private static final int MAX_STRING_LOOKAHEAD_SIZE = 512;   // temporary
    private static final int MAX_SPACE_LOOKAHEAD_SIZE = 32;     // temporary
    // Note that the charqueue size should be bigger than the longest string in the file + reader_buff_size
    //       (if we use "look ahead" for string parsing)!!!!
    // The parser/tokenizer fails when it encounters a string longer than that.
    // We cannot obviously use arbitrarily long char queue size, and
    // unfortunately, there is no limit in the size of a string in JSON,
    //    which makes it possible that the uncilient parser can always fail, potentially...
    private static final int CHARQUEUE_SIZE = 4096;     // temporary
    // Note that CHARQUEUE_SIZE - delta >= READER_BUFF_SIZE
    //     where delta is "false".length,
    //          or, more precisely the max length of peekChars(len).
    //          or, if we use string lookahead, it should be the size of the longest string rounded up to MAX_STRING_LOOKAHEAD_SIZE multiples...
    private static final int READER_BUFF_SIZE = 1024;   // temporary
    private static final int HEAD_TRACE_LENGTH = 25;     // temporary
    // ...

    private Reader reader;
    private int curReaderIndex = 0;       // global var to keep track of the reader reading state.
    private boolean readerEOF = false;     // true does not mean we are done, because we are using buffer.
    private JsonToken curToken = null;
    private JsonToken nextToken = null;   // or, TOKEN_EOF ???
    private JsonToken nextNextToken = null;   // ??? TBD: ...
//    private CharacterQueue charQueue = null;
    // Note that charQueue is class-wide global variable.
    private final CharQueue charQueue;
    // If true, use "look ahead" algorithms.
    // private boolean lookAheadParsing;

    // Ctor's
    public MiniClientJsonTokenizer(String str)
    {
        this(new StringReader(str));
    }
    public MiniClientJsonTokenizer(Reader reader)
    {
        this(reader, CHARQUEUE_SIZE);
    }
    public MiniClientJsonTokenizer(Reader reader, int charQueueSize)
    {
        // Reader cannot cannot be null.
        this.reader = reader;
        this.charQueue = new CharQueue(charQueueSize);
        // lookAheadParsing = true;
        
        // For subclasses
        init();
    }
    
    // Having multiple ctor's is a bit inconvenient.
    // Put all init routines here.
    protected void init()
    {
        // Override this in subclasses.
    }
    
    
    // Make tokenizer re-usable through reset().
    // TBD: Need to verify this....

    public void reset(String str)
    {
        reset(new StringReader(str));
    }
    public void reset(Reader reader)
    {
        // This is essentially a copy of ctor.
        
        // Reader cannot cannot be null.
        this.reader = reader;
        this.charQueue.clear();
        
        // Reset the "current state" vars.
        readerEOF = false;
        curToken = null;
        nextToken = null;

        // No need to call this... ???
        // init();
    }
    
    
//    public boolean isLookAheadParsing()
//    {
//        return lookAheadParsing;
//    }

    
    // TraceableJsonTokenizer interface.
    // These are primarily for debugging purposes...


    public String peekCharsAsString(int length)
    {
        char[] c = peekCharStream(length);
        if(c != null) {
            return new String(c);
        } else {
            return "";   // ????
        }
    }
    public char[] peekCharStream(int length)
    {
        char[] c = null;
        try {
            c = peekChars(length);
        } catch (MiniClientJsonException e) {
            log.log(Level.WARNING, "Failed to peek char stream: length = " + length, e);
        }
        return c;
    }
    public char[] peekCharStream()
    {
        return peekCharStream(HEAD_TRACE_LENGTH);
    }
 
    
    // TBD:
    // These methods really need to be synchronized.
    // ...
    
    @Override
    public boolean hasMore() throws MiniClientJsonException
    {
        if(nextToken == null) {
            nextToken = prepareNextToken();
        }

        if(nextToken == null || TOKEN_EOF.equals(nextToken)) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public JsonToken next() throws MiniClientJsonException
    {
        if(nextToken == null) {
            nextToken = prepareNextToken();
        }
        curToken = nextToken;
        nextToken = null;
 
        return curToken;
    }

    @Override
    public JsonToken peek() throws MiniClientJsonException
    {
        if(nextToken != null) {
            return nextToken;
        }
        
        nextToken = prepareNextToken();
        return nextToken;
    }

    // temporary
    // Does this save anything compared to next();peek(); ????
    //    (unless we can do prepareNextTwoTokens() .... ? )
    // Remove the next token (and throw away),
    // and return the next token (without removing it).
    public JsonToken nextAndPeek() throws MiniClientJsonException
    {
        if(nextToken == null) {
            curToken = prepareNextToken();
        } else {
            curToken = nextToken;
        }
        nextToken = prepareNextToken();
        return nextToken;
    }

    
    // Note that this method is somewhat unusual in that it cannot be called arbitrarily.
    // This method changes the internal state by changing the charQueue.
    // This should be called by a certain select methods only!!!
    private JsonToken prepareNextToken() throws MiniClientJsonException
    {
        if(nextToken != null) {
            // ???
            return nextToken;
        }

        JsonToken token = null;
        char ch;
        // ch has been peeked, but not popped.
        // ...
        // "Look ahead" version should be just a bit faster....
        ch = gobbleUpSpaceLookAhead();
        // ch = gobbleUpSpace();
        // ....

        // Create a JsonToken and,
        // reset the curToken.
        switch(ch) {
        case Symbols.COMMA:
        case Symbols.COLON:
        case Symbols.LSQUARE:
        case Symbols.LCURLY:
        case Symbols.RSQUARE:
        case Symbols.RCURLY:
            token = TokenPool.getSymbolToken(ch);
            // nextChar();   // Consume the current token.
            skipCharNoCheck();   // Consume the current token.
            // nextToken = null;
            break;
        case Symbols.NULL_START:
        case Symbols.NULL_START_UPPER:
            token = doNullLiteral();
            break;
        case Symbols.TRUE_START:
        case Symbols.TRUE_START_UPPER:
            token = doTrueLiteral();
            break;
        case Symbols.FALSE_START:
        case Symbols.FALSE_START_UPPER:
            token = doFalseLiteral();
            break;
        case Symbols.DQUOTE:
            token = doString();
            break;
        case 0:
            // ???
            token = TokenPool.TOKEN_EOF;
            // nextChar();   // Consume the current token.
            skipCharNoCheck();   // Consume the current token.
            break;
        default:
            if(Symbols.isStartingNumber(ch)) {
                // log.warning(">>>>>>>>>>>>>>>>>>>>>>>>>> ch = " + ch);
                token = doNumber();
                // log.warning(">>>>>>>>>>>>>>>>>>>>>>>>>> number token = " + token);
            } else {
                throw new MiniClientJsonException("Invalid symbol encountered: ch = " + ch);
            }
            break;
        }

        return token;
    }

    
    private char gobbleUpSpace()
    {
        char c = 0;
        try {
            c = peekChar();
            //while(c != 0 && Character.isSpaceChar(c)) {  // ???  -> this doesn't seem to work....
            // while(c != 0 && Character.isWhitespace(c)  ) {  // ???
            while(c != 0 && CharacterUtil.isWhitespace(c)  ) {  // ???
                // nextChar();   // gobble up space.
                // c = peekChar();
                c = skipAndPeekChar();
            }
        } catch(MiniClientJsonException e) {
            // ????
            if(log.isLoggable(Level.INFO)) log.log(Level.INFO, "Failed to consume space.", e);
            c = 0;
        }
        return c;
    }

    // Returns the next peeked character.
    // Return value of 0 means we have reached the end of the json string.
    // TBD: use "look ahead" implementation similar to readString() ????
    // Note that this is effective only for "formatted" JSON with lots of consecutive spaces...
    private char gobbleUpSpaceLookAhead()
    {
        char c = 0;
        try {
            c = peekChar();
            // if(Character.isWhitespace(c)) {
            if(CharacterUtil.isWhitespace(c)) {
                // skipCharNoCheck();
                c = skipAndPeekChar();
                
                // Spaces tend appear together.
                // if(Character.isWhitespace(c)) {
                if(CharacterUtil.isWhitespace(c)) {
                    int chunkLength;
                    CyclicCharArray charArray = peekCharsInQueue(MAX_SPACE_LOOKAHEAD_SIZE);
                    // if(charArray == null || (chunkLength = charArray.getLength()) == 0) {
                    //     return c;
                    // }
                    chunkLength = charArray.getLength();
    
                    int chunkCounter = 0;
                    int totalLookAheadLength = 0;
                    c = charArray.getChar(0);
                    // while((chunkCounter < chunkLength - 1) && Character.isWhitespace(c) ) {
                    while((chunkCounter < chunkLength - 1) && CharacterUtil.isWhitespace(c) ) {
                        ++chunkCounter;
    
                        if(chunkCounter >= chunkLength - 1) {
                            totalLookAheadLength += chunkCounter;
                            chunkCounter = 0;   // restart a loop.
    
                            charArray = peekCharsInQueue(totalLookAheadLength, MAX_SPACE_LOOKAHEAD_SIZE);
                            if(charArray == null || (chunkLength = charArray.getLength()) == 0) {
                                break;
                            }
                        }
                        c = charArray.getChar(chunkCounter);
                    }
                    totalLookAheadLength += chunkCounter;
                    skipChars(totalLookAheadLength);
                    c = peekChar();
                }
            }
        } catch(MiniClientJsonException e) {
            // ????
            if(log.isLoggable(Level.INFO)) log.log(Level.INFO, "Failed to consume space.", e);
            c = 0;
        }
        return c;
    }

    private JsonToken doNullLiteral() throws MiniClientJsonException
    {
        JsonToken token = null;
        int length = Literals.NULL_LENGTH;
        // char[] c = nextChars(length);
        CyclicCharArray c = nextCharsInQueue(length);
        if(LiteralUtil.isNull(c)) {
            token = TokenPool.TOKEN_NULL;
            // nextToken = null;
        } else {
            // throw new JsonException("Unexpected string: " + Arrays.toString(c), tailCharStream());
            throw new MiniClientJsonException("Unexpected string: ");
        }        
        return token;
    }
    private JsonToken doTrueLiteral() throws MiniClientJsonException
    {
        JsonToken token = null;
        int length = Literals.TRUE_LENGTH;
        // char[] c = nextChars(length);
        CyclicCharArray c = nextCharsInQueue(length);
        if(LiteralUtil.isTrue(c)) {
            token = TokenPool.TOKEN_TRUE;
            // nextToken = null;
        } else {
            // throw new JsonException("Unexpected string: " + Arrays.toString(c), tailCharStream());
            throw new MiniClientJsonException("Unexpected string: ");
        }        
        return token;
    }
    private JsonToken doFalseLiteral() throws MiniClientJsonException
    {
        JsonToken token = null;
        int length = Literals.FALSE_LENGTH;
        // char[] c = nextChars(length);
        CyclicCharArray c = nextCharsInQueue(length);
        if(LiteralUtil.isFalse(c)) {
            token = TokenPool.TOKEN_FALSE;
            // nextToken = null;
        } else {
            // throw new JsonException("Unexpected string: " + Arrays.toString(c), tailCharStream());
            throw new MiniClientJsonException("Unexpected string: ");
        }   
        return token;
    }
    
    // Note that there is no "character".
    // Character is a single letter string.
    private JsonToken doString() throws MiniClientJsonException
    {
        JsonToken token = null;
        String value;
        // ....
        // value = readString();
        // Note that this will fail if we encounter a looooong string.
        //     See the comment below. We try at least once with readString() version...
        value = readStringWithLookAhead();
        // ....
        token = TokenPool.getInstance().getToken(TokenTypes.STRING, value);
        // nextToken = null;
        return token;
    }
    
    private String readString() throws MiniClientJsonException
    {
        // Note that we may have already "consumed" the beginning \" if we are calling this from readStringWithLookAhead()...
        // So, the following does not work....

//        // char c = nextChar();
//        char c = nextCharNoCheck();
//        if(c == 0 || c != Symbols.DQUOTE) {
//            // This cannot happen.
//            throw new MiniClientJsonException("Expecting String. Invalid token encountered: c = " + c);
//        }

        StringBuilder sb = new StringBuilder();

        char c = peekChar();
        if(c == 0) {
            // This cannot happen.
            throw new MiniClientJsonException("Expecting String. Invalid token encountered: c = " + c);
        } else if(c == Symbols.DQUOTE) {
            // consume the leading \".
            // c = nextCharNoCheck();
            skipCharNoCheck();
            // sb.append(c);   // No append: Remove the leading \".
        } else {
            // We are already at the beginning of the string.
            // proceed.
        }

        boolean escaped = false;
        char d = peekChar();
        while(d != 0 && (escaped == true || d != Symbols.DQUOTE )) {
            // d = nextChar();
            d = nextCharNoCheck();
            if(escaped == false && d == Symbols.BACKSLASH) {
                escaped = true;
                // skip
            } else {
                if(escaped == true) {
                    if(d == Symbols.UNICODE_PREFIX) {
                        // char[] hex = nextChars(4);
                        CyclicCharArray hex = nextCharsInQueue(4);
                        // TBD: validate ??
                        
                        try {
                            // ????
                            // sb.append(Symbols.BACKSLASH).append(d).append(hex);
                            char u = UnicodeUtil.getUnicodeChar(hex);
                            if(u != 0) {
                                sb.append(u);
                            } else {
                                // ????
                            }
                        } catch(Exception e) {
                            throw new MiniClientJsonException("Invalid unicode char: hex = " + hex.toString(), e);
                        }
                    } else {
                        if(Symbols.isEscapableChar(d)) {
                            // TBD:
                            // Newline cannot be allowed within a string....
                            // ....
                            char e = Symbols.getEscapedChar(d);
                            if(e != 0) {
                                sb.append(e);
                            } else {
                                // This cannot happen.
                            }
                        } else {
                            // error?
                            throw new MiniClientJsonException("Invalid escaped char: d = \\" + d);
                        }
                    }
                    // toggle the flag.
                    escaped = false;
                } else {
                    
                    // TBD:
                    // Exclude control characters ???
                    // ...
                    
                    sb.append(d);
                }
            }
            d = peekChar();
        }
        if(d == Symbols.DQUOTE) {
            // d = nextChar();
            skipCharNoCheck();
            // sb.append(d);  // No append: Remove the trailing \".
        } else {
            // end of the json string.
            // error???
            // return null;
        }
        
        return sb.toString();
    }

    // Note:
    // This will cause parse failing
    //     if the longest string in JSON is longer than (CHARQUEUE_SIZE - READER_BUFF_SIZE)
    //     because forward() will fail.
    // TBD:
    // There might be bugs when dealing with short strings, or \\u escaped unicodes at the end of a json string
    // ...
    private String readStringWithLookAhead() throws MiniClientJsonException
    {
        // char c = nextChar();
        char c = nextCharNoCheck();
        if(c == 0 || c != Symbols.DQUOTE) {
            // This cannot happen.
            throw new MiniClientJsonException("Expecting String. Invalid token encountered: c = " + c);
        }
        StringBuilder sb = new StringBuilder();
        // sb.append(c);   // No append: Remove the leading \".
        
        boolean escaped = false;
        
        
        int chunkLength;
        CyclicCharArray charArray = peekCharsInQueue(MAX_STRING_LOOKAHEAD_SIZE);
        if(charArray == null || (chunkLength = charArray.getLength()) == 0) {
            // ????
            throw new MiniClientJsonException("String token terminated unexpectedly.");
        }
        boolean noMoreCharsInQueue = false;
        if(chunkLength < MAX_STRING_LOOKAHEAD_SIZE) {
            noMoreCharsInQueue = true;
        }
        boolean needMore = false;
        int chunkCounter = 0;
        int totalLookAheadLength = 0;
        char d = charArray.getChar(0);
        // log.warning(">>>>>>>>>>>>>>>>>> d = " + d);
        // log.warning(">>>>>>>>>>>>>>>>>> chunkLength = " + chunkLength);
        while((chunkCounter < chunkLength - 1) &&    // 6 for "\\uxxxx". 
                d != 0 && 
                (escaped == true || d != Symbols.DQUOTE )) {
            // d = charArray.getChar(++chunkCounter);
            ++chunkCounter;
            
            // log.warning(">>>>>>>>>>>>>>>>>> d = " + d);
            
            if(escaped == false && d == Symbols.BACKSLASH) {
                escaped = true;
                // skip
            } else {
                if(escaped == true) {
                    if(d == Symbols.UNICODE_PREFIX) {
                        if(chunkCounter < chunkLength - 4) {
                            char[] hex = charArray.getChars(chunkCounter, 4);
                            chunkCounter += 4;
                            
                            try {
                                // ????
                                // sb.append(Symbols.BACKSLASH).append(d).append(hex);
                                char u = UnicodeUtil.getUnicodeChar(hex);
                                if(u != 0) {
                                    sb.append(u);
                                } else {
                                    // ????
                                }
                            } catch(Exception e) {
                                throw new MiniClientJsonException("Invalid unicode char: hex = " + Arrays.toString(hex), e);
                            }
                        } else {
                            if(noMoreCharsInQueue == false) {
                                needMore = true;
                                chunkCounter -= 2;     // Reset the counter backward for "\\u".
                            } else {
                                // error
                                throw new MiniClientJsonException("Invalid unicode char.");
                            }
                        }
                    } else {
                        if(Symbols.isEscapableChar(d)) {
                            // TBD:
                            // Newline cannot be allowed within a string....
                            // ....
                            char e = Symbols.getEscapedChar(d);
                            if(e != 0) {
                                sb.append(e);
                            } else {
                                // This cannot happen.
                            }
                        } else {
                            // error?
                            throw new MiniClientJsonException("Invalid escaped char: d = \\" + d);
                        }
                    }
                    // toggle the flag.
                    escaped = false;
                } else {
                    
                    // TBD:
                    // Exclude control characters ???
                    // ...
                    
                    sb.append(d);
                }
            }
            
            if((noMoreCharsInQueue == false) && (needMore || chunkCounter >= chunkLength - 1)) {
                totalLookAheadLength += chunkCounter;
                chunkCounter = 0;   // restart a loop.
                needMore = false;
                // log.warning(">>>>>>>>>>>>>>>>>>>>>> addAll() totalLookAheadLength = " + totalLookAheadLength);

                try {
                    charArray = peekCharsInQueue(totalLookAheadLength, MAX_STRING_LOOKAHEAD_SIZE);
                } catch(MiniClientJsonException e) {
                    // Not sure if this makes sense....
                    // but since this error might have been due to the fact that we have encountered a looooong string,
                    // Try again???
                    // ...
                    // Note that this applies one, this particular, string only.
                    // Next time when we encounter a long string, 
                    // this may be invoked again....
                    // ....
                    // We should be careful not to get into the infinite loop....
                    log.warning("String token might have been too long. Trying again with no look-ahead readString().");

                    // Reset the buffer (peek() status) ????, and call the non "look ahead" version...
                    return readString();   // Is this starting from the beginning???
                    // ...
                }
                if(charArray == null || (chunkLength = charArray.getLength()) == 0) {
                    // ????
                    throw new MiniClientJsonException("String token terminated unexpectedly.");
                }
                if(chunkLength < MAX_STRING_LOOKAHEAD_SIZE) {
                    noMoreCharsInQueue = true;
                }
            }
            
            d = charArray.getChar(chunkCounter);
        }
        totalLookAheadLength += chunkCounter;
        skipChars(totalLookAheadLength);
        d = peekChar();

        if(d == Symbols.DQUOTE) {
            // d = nextChar();
            skipCharNoCheck();
            // sb.append(d);  // No append: Remove the trailing \".
        } else {
            // end of the json string.
            // error???
            // return null;
        }
        
        return sb.toString();
    }
    
    
    private JsonToken doNumber() throws MiniClientJsonException
    {
        JsonToken token = null;
        Number value = readNumber();
        token = TokenPool.getInstance().getToken(TokenTypes.NUMBER, value);
        // nextToken = null;
        return token;
    }

    // Need a better way to do this ....
    private Number readNumber() throws MiniClientJsonException
    {
        // char c = nextChar();
        char c = nextCharNoCheck();
        if(! Symbols.isStartingNumber(c)) {
            throw new MiniClientJsonException("Expecting a number. Invalid symbol encountered: c = " + c);
        }

        if(c == Symbols.PLUS) {
            // remove the leading +.
            c = nextChar();
        }

        StringBuilder sb = new StringBuilder();

        if(c == Symbols.MINUS) {
            sb.append(c);
            c = nextChar();
        }

        boolean periodRead = false;
        if(c == Symbols.PERIOD) {
            periodRead = true;
            sb.append("0.");
        } else {
            // Could be a number, nothing else.
            if(c == '0') {
                char c2 = peekChar();
                // This does not work because the number happens to be just zero ("0").
                // if(c2 != Symbols.PERIOD) {
                //     throw new JsonException("Invalid number: c = " + sb.toString() + c + c2, tailCharStream());
                // }
                // This should be better.
                // zero followed by other number is not allowed.
                if(Character.isDigit(c2)) {
                    throw new MiniClientJsonException("Invalid number: c = " + sb.toString() + c + c2);
                }
                sb.append(c);
                if(c2 == Symbols.PERIOD) {
                    periodRead = true;
                    // sb.append(nextChar());
                    sb.append(nextCharNoCheck());
                }
            } else {
                sb.append(c);
            }
        }
        
        boolean exponentRead = false;
        
        char d = peekChar();
        while(d != 0 && (Character.isDigit(d) || 
                (periodRead == false && d == Symbols.PERIOD) ||
                (exponentRead == false && Symbols.isExponentChar(d))
                )) {
            // sb.append(nextChar());
            sb.append(nextCharNoCheck());
            if(d == Symbols.PERIOD) {
                periodRead = true;
            }
            if(Symbols.isExponentChar(d)) {
                char d2 = peekChar();
                if(d2 == Symbols.PLUS || d2 == Symbols.MINUS || Character.isDigit(d2)) {
                    // sb.append(nextChar());
                    sb.append(nextCharNoCheck());
                } else {
                    throw new MiniClientJsonException("Invalid number: " + sb.toString() + d2);
                }
                exponentRead = true;
            }
            d = peekChar();
        }
        if(d == 0) {
            // end of the json string.
            // ????
            // throw new JsonException("Invalid number: " + sb.toString(), tailCharStream());
        } else {
            // sb.append(nextChar());
        }
        
        String str = sb.toString();
        
        Number number = null;
        try {
            if(str.contains(".")) {
                double x = Double.parseDouble(str);
                // number = BigDecimal.valueOf(x);
                number = x;
            } else {
                long y = Long.parseLong(str);
                // number = BigDecimal.valueOf(y);
                number = y;
            }
        // } catch(NumberFormatException e) {
        } catch(Exception e) {
            // ???
            throw new MiniClientJsonException("Invalid number encountered: str = " + str, e);
        }
        return number;
    }

    
    // because we called peekChar() already,
    //      no need for check error conditions.
    private char nextCharNoCheck() throws MiniClientJsonException
    {
        char ch = charQueue.poll();
        return ch;
    }
    private void skipCharNoCheck() throws MiniClientJsonException
    {
        charQueue.skip();
    }

    private char nextChar() throws MiniClientJsonException
    {
        if(charQueue.isEmpty()) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(charQueue.isEmpty()) {
            return 0;   // ???
            // throw new JsonException("There is no character in the buffer.");
        }
        char ch = charQueue.poll();
        return ch;
    }
    private char[] nextChars(int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        char[] c = null;
        if(charQueue.size() < length) {
            c = charQueue.poll(charQueue.size());
            // throw new JsonException("There is not enough characters in the buffer. length = " + length);
        }
        c = charQueue.poll(length);
        return c;
    }
    private CyclicCharArray nextCharsInQueue(int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        CyclicCharArray charArray = null;
        if(charQueue.size() < length) {
            charArray = charQueue.pollBuffer(charQueue.size());
            // throw new JsonException("There is not enough characters in the buffer. length = " + length);
        }
        charArray = charQueue.pollBuffer(length);
        return charArray;
    }
    
    private void skipChars(int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        charQueue.skip(length);
    }

    
    

    // Note that peekChar() and peekChars() are "idempotent". 
    private char peekChar() throws MiniClientJsonException
    {
        if(charQueue.isEmpty()) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(charQueue.isEmpty()) {
            return 0;
            // throw new JsonException("There is no character in the buffer.");
        }
        return charQueue.peek();
    }
    private char[] peekChars(int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(charQueue.size() < length) {
            return charQueue.peek(charQueue.size());
            // throw new JsonException("There is not enough characters in the buffer. length = " + length);
        }
        return charQueue.peek(length);
    }
    private CyclicCharArray peekCharsInQueue(int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(charQueue.size() < length) {
            return charQueue.peekBuffer(charQueue.size());
            // throw new JsonException("There is not enough characters in the buffer. length = " + length);
        }
        return charQueue.peekBuffer(length);
    }
    private CyclicCharArray peekCharsInQueue(int offset, int length) throws MiniClientJsonException
    {
        // assert length > 0
        if(charQueue.size() < offset + length) {
            if(readerEOF == false) {
                try {
                    forward();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(charQueue.size() < offset + length) {
            return charQueue.peekBuffer(offset, charQueue.size() - offset);
            // throw new JsonException("There is not enough characters in the buffer. length = " + length);
        }
        return charQueue.peekBuffer(offset, length);
    }

    
    // Poll next char (and gobble up),
    // and return the next char (without removing it)
    private char skipAndPeekChar() throws MiniClientJsonException
    {
        int qSize = charQueue.size();
        if(qSize < 2) {
            if(readerEOF == false) {
                try {
                    forward();
                    qSize = charQueue.size();
                } catch (IOException e) {
                    // ???
                    throw new MiniClientJsonException("Failed to forward character stream.", e);
                }
            }
        }
        if(qSize > 0) {
            charQueue.skip();
            if(qSize > 1) {
                return charQueue.peek();
            }
        }
        return 0;
        // throw new JsonException("There is no character in the buffer.");
    }

    
    // Read some more bytes from the reader.
    private final char[] buff = new char[READER_BUFF_SIZE];
    private void forward() throws IOException, MiniClientJsonException
    {
        if(readerEOF == false) {
            if(reader.ready()) {  // To avoid blocking
                int cnt = 0;
                try {
                    // This throws OOB excpetion at the end....
                    // cnt = reader.read(buff, curReaderIndex, READER_BUFF_SIZE);
                    cnt = reader.read(buff);
                } catch(IndexOutOfBoundsException e) {
                    // ???
                    // Why does this happen? Does it happen for StringReader only???
                    //    Does read(,,) ever return -1 in the case of StringReader ???
                    if(log.isLoggable(Level.INFO)) log.log(Level.INFO, "Looks like we have reached the end of the reader.", e);
                }
                if(cnt == -1 || cnt == 0) {
                    readerEOF = true;
                } else {
                    boolean suc = charQueue.addAll(buff, cnt);
                    if(suc) {
                        curReaderIndex += cnt;
                    } else {
                        // ???
                        throw new MiniClientJsonException("Unexpected internal error occurred. Characters were not added to CharQueue: cnt = " + cnt);
                    }
                }
            } else {
                // ????
                readerEOF = true;
                // Why does this happen ????
                // if(log.isLoggable(Level.INFO)) log.log(Level.INFO, "Looks like we have not read all characters because the reader is blocked. We'll likely have a parser error down the line.");
                // throw new JsonException("Read is blocked. Bailing out.");
            }
        }
    }
    
}
