/*
 * Copyright 2014-2020 Real Logic Limited.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.agrona.collections;

import org.agrona.generation.DoNotSub;

import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.LongConsumer;
import java.util.stream.LongStream;

/**
 * A {@link List} implementation that stores long values with the ability to not have them boxed.
 */
public class LongArrayList extends AbstractList<Long> implements List<Long>, RandomAccess, Serializable
{
    /**
     * The default value that will be used in place of null for an element.
     */
    public static final long DEFAULT_NULL_VALUE = Long.MIN_VALUE;

    /**
     * Initial capacity to which the array will be sized.
     */
    @DoNotSub public static final int INITIAL_CAPACITY = 10;

    private final long nullValue;
    @DoNotSub private int size = 0;
    private long[] elements;

    public LongArrayList()
    {
        this(INITIAL_CAPACITY, DEFAULT_NULL_VALUE);
    }

    /**
     * Construct a new list.
     *
     * @param initialCapacity for the backing array.
     * @param nullValue       to be used to represent a null element.
     */
    public LongArrayList(
        @DoNotSub final int initialCapacity,
        final long nullValue)
    {
        this.nullValue = nullValue;
        elements = new long[Math.max(initialCapacity, INITIAL_CAPACITY)];
    }

    /**
     * Create a new list that wraps an existing arrays without copying it.
     *
     * @param initialElements to be wrapped.
     * @param initialSize     of the array to wrap.
     * @param nullValue       to be used to represent a null element.
     */
    public LongArrayList(
        final long[] initialElements,
        @DoNotSub final int initialSize,
        final long nullValue)
    {
        wrap(initialElements, initialSize);
        this.nullValue = nullValue;
    }

    /**
     * Wrap an existing array without copying it.
     *
     * @param initialElements to be wrapped.
     * @param initialSize     of the array to wrap.
     * @throws IllegalArgumentException if the initialSize is is less than 0 or greater than the length of the
     * initial array.
     */
    public void wrap(
        final long[] initialElements,
        final @DoNotSub int initialSize)
    {
        if (initialSize < 0 || initialSize > initialElements.length)
        {
            throw new IllegalArgumentException(
                "illegal initial size " + initialSize + " for array length of " + initialElements.length);
        }

        elements = initialElements;
        size = initialSize;
    }

    /**
     * The value representing a null element.
     *
     * @return value representing a null element.
     */
    public long nullValue()
    {
        return nullValue;
    }

    /**
     * {@inheritDoc}
     */
    public @DoNotSub int size()
    {
        return size;
    }

    /**
     * The current capacity for the collection.
     *
     * @return the current capacity for the collection.
     */
    @DoNotSub public int capacity()
    {
        return elements.length;
    }

    /**
     * {@inheritDoc}
     */
    public void clear()
    {
        size = 0;
    }

    /**
     * Trim the underlying array to be the current size, or {@link #INITIAL_CAPACITY} if size is less.
     */
    public void trimToSize()
    {
        if (elements.length != size && elements.length > INITIAL_CAPACITY)
        {
            elements = Arrays.copyOf(elements, Math.max(INITIAL_CAPACITY, size));
        }
    }

    public Long get(
        @DoNotSub final int index)
    {
        final long value = getLong(index);

        return value == nullValue ? null : value;
    }

    /**
     * Get the element at a given index without boxing.
     *
     * @param index to get.
     * @return the unboxed element.
     */
    public long getLong(
        @DoNotSub final int index)
    {
        checkIndex(index);

        return elements[index];
    }

    public boolean add(final Long element)
    {
        return addLong(null == element ? nullValue : element);
    }

    /**
     * Add an element without boxing.
     *
     * @param element to be added.
     * @return true
     */
    public boolean addLong(final long element)
    {
        ensureCapacityPrivate(size + 1);

        elements[size] = element;
        size++;

        return true;
    }

    public void add(
        @DoNotSub final int index,
        final Long element)
    {
        addLong(index, null == element ? nullValue : element);
    }

    /**
     * Add a element without boxing at a given index.
     *
     * @param index   at which the element should be added.
     * @param element to be added.
     */
    public void addLong(
        @DoNotSub final int index,
        final long element)
    {
        checkIndexForAdd(index);

        @DoNotSub final int requiredSize = size + 1;
        ensureCapacityPrivate(requiredSize);

        if (index < size)
        {
            System.arraycopy(elements, index, elements, index + 1, size - index);
        }

        elements[index] = element;
        size++;
    }

    public Long set(
        @DoNotSub final int index,
        final Long element)
    {
        final long previous = setLong(index, null == element ? nullValue : element);

        return nullValue == previous ? null : previous;
    }

    /**
     * Set an element at a given index without boxing.
     *
     * @param index   at which to set the element.
     * @param element to be added.
     * @return the previous element at the index.
     */
    public long setLong(
        @DoNotSub final int index,
        final long element)
    {
        checkIndex(index);

        final long previous = elements[index];
        elements[index] = element;

        return previous;
    }

    /**
     * Does the list contain this element value.
     *
     * @param value of the element.
     * @return true if present otherwise false.
     */
    public boolean containsLong(final long value)
    {
        return -1 != indexOf(value);
    }

    /**
     * Index of the first element with this value.
     *
     * @param value for the element.
     * @return the index if found otherwise -1.
     */
    public @DoNotSub int indexOf(
        final long value)
    {
        for (@DoNotSub int i = 0; i < size; i++)
        {
            if (value == elements[i])
            {
                return i;
            }
        }

        return -1;
    }

    /**
     * Index of the last element with this value.
     *
     * @param value for the element.
     * @return the index if found otherwise -1.
     */
    public @DoNotSub int lastIndexOf(
        final long value)
    {
        for (@DoNotSub int i = size - 1; i >= 0; i--)
        {
            if (value == elements[i])
            {
                return i;
            }
        }

        return -1;
    }

    /**
     * Remove at a given index.
     *
     * @param index of the element to be removed.
     * @return the existing value at this index.
     */
    public Long remove(
        @DoNotSub final int index)
    {
        checkIndex(index);

        final long value = elements[index];

        @DoNotSub final int moveCount = size - index - 1;
        if (moveCount > 0)
        {
            System.arraycopy(elements, index + 1, elements, index, moveCount);
        }

        size--;

        return value;
    }

    /**
     * Removes element at index, but instead of copying all elements to the left,
     * it replaces the item in the slot with the last item in the list. This avoids the copy
     * costs at the expense of preserving list order. If index is the last element it is just removed.
     *
     * @param index of the element to be removed.
     * @return the existing value at this index.
     * @throws IndexOutOfBoundsException if index is out of bounds.
     */
    public long fastUnorderedRemove(
        @DoNotSub final int index)
    {
        checkIndex(index);

        final long value = elements[index];
        elements[index] = elements[--size];

        return value;
    }

    /**
     * Remove the first instance of a value if found in the list.
     *
     * @param value to be removed.
     * @return true if successful otherwise false.
     */
    public boolean removeLong(final long value)
    {
        @DoNotSub final int index = indexOf(value);
        if (-1 != index)
        {
            remove(index);

            return true;
        }

        return false;
    }

    /**
     * Remove the first instance of a value if found in the list and replaces it with the last item
     * in the list. This saves a copy down of all items at the expense of not preserving list order.
     *
     * @param value to be removed.
     * @return true if successful otherwise false.
     */
    public boolean fastUnorderedRemoveLong(final long value)
    {
        @DoNotSub final int index = indexOf(value);
        if (-1 != index)
        {
            elements[index] = elements[--size];

            return true;
        }

        return false;
    }

    /**
     * Push an element onto the end of the array like a stack.
     *
     * @param element to be pushed onto the end of the array.
     */
    public void pushLong(final long element)
    {
        ensureCapacityPrivate(size + 1);

        elements[size] = element;
        size++;
    }

    /**
     * Pop a value off the end of the array as a stack operation.
     *
     * @return the value at the end of the array.
     * @throws NoSuchElementException if the array is empty.
     */
    public long popLong()
    {
        if (isEmpty())
        {
            throw new NoSuchElementException();
        }

        return elements[--size];
    }

    /**
     * For each element in order provide the long value to a {@link LongConsumer}.
     *
     * @param action to be taken for each element.
     */
    public void forEachOrderedLong(final LongConsumer action)
    {
        for (@DoNotSub int i = 0; i < size; i++)
        {
            action.accept(elements[i]);
        }
    }

    /**
     * Create a {@link LongStream} over the elements of underlying array.
     *
     * @return a {@link LongStream} over the elements of underlying array.
     */
    public LongStream longStream()
    {
        return Arrays.stream(elements, 0, size);
    }

    /**
     * Create a new array that is a copy of the elements.
     *
     * @return a copy of the elements.
     */
    public long[] toLongArray()
    {
        return Arrays.copyOf(elements, size);
    }

    /**
     * Create a new array that is a copy of the elements.
     *
     * @param dst destination array for the copy if it is the correct size.
     * @return a copy of the elements.
     */
    public long[] toLongArray(final long[] dst)
    {
        if (dst.length == size)
        {
            System.arraycopy(elements, 0, dst, 0, dst.length);
            return dst;
        }
        else
        {
            return Arrays.copyOf(elements, size);
        }
    }

    /**
     * Ensure the backing array has a required capacity.
     *
     * @param requiredCapacity for the backing array.
     */
    public void ensureCapacity(@DoNotSub final int requiredCapacity)
    {
        ensureCapacityPrivate(Math.max(requiredCapacity, INITIAL_CAPACITY));
    }

    public boolean equals(final LongArrayList that)
    {
        if (that == this)
        {
            return true;
        }

        boolean isEqual = false;

        if (this.size == that.size)
        {
            isEqual = true;

            for (@DoNotSub int i = 0; i < size; i++)
            {
                final long thisValue = this.elements[i];
                final long thatValue = that.elements[i];

                if (thisValue != thatValue)
                {
                    if (thisValue != this.nullValue || thatValue != that.nullValue)
                    {
                        isEqual = false;
                        break;
                    }
                }
            }
        }

        return isEqual;
    }

    /**
     * {@inheritDoc}
     */
    public boolean equals(final Object other)
    {
        if (other == this)
        {
            return true;
        }

        boolean isEqual = false;

        if (other instanceof LongArrayList)
        {
            return equals((LongArrayList)other);
        }
        else if (other instanceof List)
        {
            final List<?> that = (List<?>)other;

            if (this.size == that.size())
            {
                isEqual = true;
                @DoNotSub int i = 0;

                for (final Object o : that)
                {
                    if (o == null || o instanceof Long)
                    {
                        final Long thisValue = get(i++);
                        final Long thatValue = (Long)o;

                        if (Objects.equals(thisValue, thatValue))
                        {
                            continue;
                        }
                    }

                    isEqual = false;
                    break;
                }
            }
        }

        return isEqual;
    }

    /**
     * {@inheritDoc}
     */
    @DoNotSub public int hashCode()
    {
        @DoNotSub int hashCode = 1;
        for (@DoNotSub int i = 0; i < size; i++)
        {
            final long value = elements[i];
            hashCode = 31 * hashCode + (value == nullValue ? 0 : Long.hashCode(value));
        }

        return hashCode;
    }

    /**
     * {@inheritDoc}
     */
    public void forEach(final Consumer<? super Long> action)
    {
        for (@DoNotSub int i = 0; i < size; i++)
        {
            final long value = elements[i];
            action.accept(value != nullValue ? value : null);
        }
    }

    /**
     * Iterate over the collection without boxing.
     *
     * @param action to be taken for each element.
     */
    public void forEachLong(final LongConsumer action)
    {
        for (@DoNotSub int i = 0; i < size; i++)
        {
            action.accept(elements[i]);
        }
    }

    /**
     * {@inheritDoc}
     */
    public String toString()
    {
        final StringBuilder sb = new StringBuilder();
        sb.append('[');

        for (@DoNotSub int i = 0; i < size; i++)
        {
            final long value = elements[i];
            sb.append(value != nullValue ? value : null).append(", ");
        }

        if (sb.length() > 1)
        {
            sb.setLength(sb.length() - 2);
        }

        sb.append(']');

        return sb.toString();
    }

    private void ensureCapacityPrivate(@DoNotSub final int requiredCapacity)
    {
        @DoNotSub final int currentCapacity = elements.length;
        if (requiredCapacity > currentCapacity)
        {
            if (requiredCapacity > ArrayUtil.MAX_CAPACITY)
            {
                throw new IllegalStateException("max capacity: " + ArrayUtil.MAX_CAPACITY);
            }

            @DoNotSub int newCapacity = currentCapacity > INITIAL_CAPACITY ? currentCapacity : INITIAL_CAPACITY;

            while (newCapacity < requiredCapacity)
            {
                newCapacity = newCapacity + (newCapacity >> 1);

                if (newCapacity < 0 || newCapacity >= ArrayUtil.MAX_CAPACITY)
                {
                    newCapacity = ArrayUtil.MAX_CAPACITY;
                }
            }

            final long[] newElements = new long[newCapacity];
            System.arraycopy(elements, 0, newElements, 0, currentCapacity);
            elements = newElements;
        }
    }

    private void checkIndex(@DoNotSub final int index)
    {
        if (index >= size || index < 0)
        {
            throw new IndexOutOfBoundsException("index=" + index + " size=" + size);
        }
    }

    private void checkIndexForAdd(@DoNotSub final int index)
    {
        if (index > size || index < 0)
        {
            throw new IndexOutOfBoundsException("index=" + index + " size=" + size);
        }
    }
}
