package org.aika.corpus;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.aika.network.neuron.Activation;

import java.util.*;

/**
 * The <code>Document</code> class represents a single document within a corpus of documents. A set of documents can be
 * either used to train a new model or a document can be processed and annotated by an already trained model.
 * The <code>Document</code> class contains the actual text, a set of initial annotations, the option lattice containing
 * all the possible interpretations of this document, a set of option conflicts and a ranked list of weighted options.
 *
 * @author Lukas Molzberger
 */
public class Document {

    private String content;

    public int primOptionIdCounter = 0;
    public int optionIdCounter = 1;

    public Option top = new Option(this, true, -1, Integer.MAX_VALUE, -1, -1, Integer.MAX_VALUE);
    public Option bottom = new Option(this, false, -1, 0, -1, -1, 0);

    public ExpandNode selectedExpandNode = null;
    public Option selectedOption = null;
    public long selectedMark = -1;

    public Range topRange = new Range(this, true, Range.toSeg(0, Integer.MAX_VALUE));
    public Range bottomRange = new Range(this, false, new ArrayList<>());


    public Map<List<int[]>, Range> ranges = new TreeMap<>(Range.SEGMENT_COMPARATOR);


    public Document(String content) {
        this.content = content;
        top.negation = bottom;
        bottom.negation = top;
        Option.addLink(bottom, top);

        topRange.negation = bottomRange;
        bottomRange.negation = topRange;

        Range.initLattice(this);
    }


    public static Document create(String content) {
        return new Document(content);
    }


    public boolean isSelected(Activation act) {
        if(!selectedOption.contains(act.key.o)) return false;
        if(act.newOption != null && !selectedOption.contains(act.newOption)) return false;

        return true;
    }


    public String getContent() {
        return content;
    }


    public int length() {
        return content.length();
    }


    public String toString() {
		return content;
	}


    public String conflictsToString() {
        HashSet<Option> conflicts = new HashSet<>();
        top.collectConflicts(conflicts, Option.visitedCounter++);

        StringBuilder sb = new StringBuilder();
        sb.append("Conflicts:\n");
        for(Option n: conflicts) {
            sb.append(n.conflicts.primaryToString());
        }
        sb.append("\n");
        return sb.toString();
    }


    public String selectedOptionsToString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Selected Options:\n");
        sb.append(selectedOption.toString());
        sb.append("\n");
        return sb.toString();
    }


    public boolean contains(Range r) {
        return r.getBegin() >= 0 && r.getEnd() <= length();
    }
}
