package org.aika.corpus;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


import org.aika.network.neuron.Activation;
import org.aika.network.neuron.Activation.Key;

import java.util.*;

/**
 *
 * @author Lukas Molzberger
 */
public class Range implements Comparable<Range> {

    public static final Range MIN = new Range(null, false, toSeg(0, 0));
    public static final Range MAX = new Range(null, true, toSeg(Integer.MAX_VALUE, Integer.MAX_VALUE));

    public final boolean inv;
    List<int[]> segments;
    public final int length;
    public boolean isPrimitive;

    public final Document doc;

    public boolean isRemoved;
    public int removedId;
    public static int removedIdCounter = 1;

    public ArrayList<Range> parents = new ArrayList<>();
    public ArrayList<Range> children = new ArrayList<>();

    int allParents = 0;
    int allChildren = 0;

    public Range negation;

    private static final Comparator<Key> COMP = new Comparator<Key>() {
        @Override
        public int compare(Key k1, Key k2) {
            int r = k1.n.compareTo(k2.n);
            if(r != 0) return r;
            r = Integer.compare(k1.rid, k2.rid);
            if(r != 0) return r;
            r = k1.o.compareTo(k2.o);
            if(r != 0) return r;
            r = Integer.compare(k1.fired, k2.fired);
            if(r != 0) return r;
            return Integer.compare(k1.id, k2.id);
        }
    };


    public static final Comparator<List<int[]>> SEGMENT_COMPARATOR = new Comparator<List<int[]>>() {
        @Override
        public int compare(List<int[]> r1, List<int[]> r2) {
            for(int i = 0; i < Math.max(r1.size(), r2.size()); i++) {
                if(i >= r1.size()) return -1;
                if(i >= r2.size()) return 1;

                int[] sa = r1.get(i);
                int[] sb = r2.get(i);
                int r = Integer.compare(sa[0], sb[0]);
                if(r != 0) return r;
                r = Integer.compare(sa[1], sb[1]);
                if(r != 0) return r;
            }
            return 0;
        }
    };

    public NavigableMap<Key, Activation> inputActivations = new TreeMap<>(COMP);
    public NavigableMap<Key, Activation> activations = new TreeMap<>(COMP);

    private long visitedContains = -1;
    private long visitedLinkRelations = -1;
    private long visitedComputeRelations = -1;
    private long visitedOptimize = -1;
    private long visitedCollect = -1;
    private long visitedCountNode = -1;
    public static long visitedCounter = 0;

    public int refCount = 0;


    public enum Relation {
        EQUALS(true),
        CONTAINS(true),
        CONTAINED_IN(true),
        OVERLAPS(false), // TODO: check if this can be optimized
        OVERLAPS_INCLUDE_ADJOINED(false),
        OVERLAPS_AFTER(false),
        BEFORE(false),
        AFTER(false),
        END_EQUALS(false),
        END_BEFORE(false),
        END_AFTER(false),
        BEGIN_EQUALS(false),
        BEGIN_BEFORE(false),
        BEGIN_AFTER(false);

        public final boolean supportsCollect;

        Relation(boolean supportsCollect) {
            this.supportsCollect = supportsCollect;
        }


        public boolean compare(List<int[]> a, List<int[]> b) {
            switch (this) {
                case EQUALS:
                    return SEGMENT_COMPARATOR.compare(a, b) == 0;
                case CONTAINS:
                    return contains(a, b);
                case CONTAINED_IN:
                    return contains(b, a);
                case OVERLAPS:
                    return overlaps(a, b, false, false);
                case OVERLAPS_INCLUDE_ADJOINED:
                    return overlaps(a, b, true, false);
                case OVERLAPS_AFTER:
                    return overlaps(a, b, true, true);
                case BEFORE:
                    return isBefore(a, b);
                case AFTER:
                    return isBefore(b, a);
                case END_BEFORE:
                    return a.get(a.size() - 1)[1] < b.get(a.size() - 1)[1];
                case END_AFTER:
                    return a.get(a.size() - 1)[1] > b.get(a.size() - 1)[1];
                case END_EQUALS:
                    return a.get(a.size() - 1)[1] == b.get(a.size() - 1)[1];
                case BEGIN_BEFORE:
                    return a.get(0)[0] < b.get(0)[0];
                case BEGIN_AFTER:
                    return a.get(0)[0] > b.get(0)[0];
                case BEGIN_EQUALS:
                    return a.get(0)[0] == b.get(0)[0];
                default:
                    return false;
            }
        }


        public int estimateNodeCount(Range r) {
            switch (this) {
                case EQUALS:
                    return 1;
                case CONTAINS:
                    return r.allParents;
                case CONTAINED_IN:
                    return r.allChildren;
                case OVERLAPS:
                    return r.allParents + r.allChildren;
                case OVERLAPS_INCLUDE_ADJOINED:
                    return r.allParents + r.allChildren;
                case OVERLAPS_AFTER:
                    return r.allParents + r.allChildren;
                default:
            }
            return Integer.MAX_VALUE;
        }
    }


    private boolean isBefore(Range b) {
        return getEnd() <= b.getBegin();
    }


    private static boolean isBefore(List<int[]> a, List<int[]> b) {
        return a.get(a.size() - 1)[1] <= b.get(0)[0];
    }


    static List<int[]> toSeg(int begin, int end) {
        ArrayList<int[]> tmp = new ArrayList<>(1);
        tmp.add(new int[] {begin, end});
        return tmp;
    }


    Range(Document doc, boolean inv, List<int[]> segments) {
        this.doc = doc;
        this.inv = inv;
        this.segments = segments;
        int l = 0;
        for(int i = 0; i < segments.size(); i++) {
            int[] seg = segments.get(i);
            l += seg[1] - seg[0];
        }
        length = l;
    }


    public static Range create(Document doc, int a, int b) {
        assert a >= 0 && a <= doc.length() && b >= 0 && b <= doc.length();
        if(a == b) return doc.bottomRange;
        int[] seg = a <= b ? new int[] {a, b} : new int[] {b, a};
        ArrayList<int[]> tmp = new ArrayList<>();
        tmp.add(seg);
        return createInternal(doc, tmp);
    }


    public static Range create(Document doc, int[] pos) {
        if(pos.length == 0) return doc.bottomRange;
        if(pos[0] == 0 && pos[1] >= Integer.MAX_VALUE) return doc.topRange;
        assert pos.length % 2 == 0;

        List<int[]> r = new ArrayList<>();

        for(int i = 0; i < pos.length; ) {
            r.add(new int[] {pos[i++], pos[i++]});
        }
        return createInternal(doc, mergeAdjoined(r));
    }


    public static Range create(Document doc, List<int[]> r) {
        if(r.size() == 0) return doc.bottomRange;
        if(r.get(0)[0] == 0 && r.get(0)[1] == Integer.MAX_VALUE) return doc.topRange;

        return createInternal(doc, mergeAdjoined(r));
    }


    static Range createInternal(Document doc, List<int[]> pos) {
        Range r = doc.ranges.get(pos);
        if(r != null) return r;

        ArrayList<Range> tmp = new ArrayList<>();
        List<int[]> uncovered = pos;
        while(uncovered.size() != 0) {
            ArrayList<Range> collected = new ArrayList<>();

            int start = uncovered.get(0)[0];
            Range startRange = doc.ranges.get(toSeg(start, start + 1));
            startRange.collectMaxCovering(collected, pos, visitedCounter++);
            for(Range tr: collected) {
                uncovered = complement(uncovered, tr.segments);
                tmp.add(tr);
            }
        }

        r = Range.add(doc, tmp.toArray(new Range[tmp.size()]));

        doc.ranges.put(pos, r);

        return r;
    }


    private boolean collectMaxCovering(ArrayList<Range> results, List<int[]> pos, long v) {
        if(visitedCollect == v) return false;
        visitedCollect = v;

        if(inv) return true;
        if(!contains(pos, segments)) return true;

        boolean flag = true;
        for(Range c: children) {
            if(!c.collectMaxCovering(results, pos, v)) flag = false;
        }

        if(flag) {
            results.add(this);
        }
        return false;
    }


    static void initLattice(Document doc) {
        Range r = new Range(doc, false, toSeg(0, doc.length()));
        doc.ranges.put(r.segments, r);
        Range nr = new Range(doc, true, invert(r.segments));
        r.negation = nr;
        nr.negation = r;
        r.initLatticeRecursiveStep();
    }


    private void initLatticeRecursiveStep() {
        if(length > 1) {
            int l = 1;
            while(l * 10 < getEnd() - getBegin()) l *= 10;

            for(int i = getBegin(); i < getEnd(); i += l) {
                Range r = new Range(doc, false, toSeg(i, Math.min(i + l, getBegin() + getEnd())));
                doc.ranges.put(r.segments, r);
                Range nr = new Range(doc, true, invert(r.segments));

                r.negation = nr;
                nr.negation = r;

                parents.add(r);
                r.children.add(this);
                negation.children.add(nr);
                nr.parents.add(negation);

                r.initLatticeRecursiveStep();
            }
        } else {
            isPrimitive = true;
            parents.add(doc.bottomRange);
            doc.bottomRange.children.add(this);
            negation.children.add(doc.bottomRange);
            doc.topRange.parents.add(negation);
        }

        countNode(false, true, visitedCounter++);
        countNode(true, true, visitedCounter++);
    }


    private void countNode(boolean dir, boolean addOrDelete, long v) {
        if(inv || visitedCountNode == v) return;
        visitedCountNode = v;

        if(dir) {
            allParents += addOrDelete ? 1 : -1;
        } else {
            allChildren += addOrDelete ? 1 : -1;
        }

        for(Range r: dir ? parents : children) {
            r.countNode(dir, addOrDelete, v);
        }
    }


    private static List<int[]> mergeAdjoined(List<int[]> r) {
        if(r.size() <= 1) {
            return r;
        }

        ArrayList<int[]> results = new ArrayList<>();
        int[] last = null;
        for(int[] s: r) {
            if(!(last == null || last[1] <= s[0])) {
                s[0] = results.get(results.size() - 1)[0];
                results.remove(results.size() - 1);
            }

            results.add(s);
            last = s;
        }
        return results;
    }


    public static Range add(Document doc, Range... input) {
        if(input.length == 0) return doc.bottomRange;

        List<int[]> segs = union(input);
        Range r = doc.ranges.get(segs);
        if(r != null) return r;

        ArrayList<Range> in = new ArrayList<>();
        int minPos = Integer.MAX_VALUE;
        int maxPos = 0;
        for(int i = 0; i < input.length; i++) {
            Range n = input[i];
            if(n == null) continue;

            assert doc == n.doc;
            assert !n.inv;
            assert !n.isRemoved;

            boolean f = true;
            for(int j = 0; j < input.length; j++) {
                Range x = input[j];
                if(x == null) continue;

                if(i != j && x.contains(n) && (x != n || i < j)) {
                    f = false;
                    break;
                }
            }
            if(f) {
                in.add(n);
                minPos = Math.min(minPos, n.getBegin());
                maxPos = Math.max(maxPos, n.getEnd());
            }
        }


        ArrayList<Range> children = new ArrayList<>();
        computeRelations(children, false, in, in, visitedCounter++);

        if(children.size() == 1) {
            Range n = children.get(0);
            if(!n.inv && SEGMENT_COMPARATOR.compare(segs, n.segments) == 0) {
                n.countRef();
                return n;
            }
        } else if(children.size() == 0) {
            children.add(doc.topRange);
        }

        ArrayList<Range> parents = new ArrayList<>();
        computeRelations(parents, true, in, children, visitedCounter++);

        Range n = new Range(doc, false, segs);
        doc.ranges.put(segs, n);

        Range nn = new Range(doc, true, invert(n.segments));
        n.negation = nn;
        nn.negation = n;

        n.linkRelations(parents, children, visitedCounter++);

        n.countRef();

        n.countNode(false, true, visitedCounter++);
        n.countNode(true, true, visitedCounter++);

        return n;
    }


    private static void computeRelations(List<Range> results, boolean dir, List<Range> input, List<Range> start, long v) {
        Range bestN = null;
        int bestLength = dir ? Integer.MAX_VALUE : 0;
        for(Range s: start) {
            if(dir == bestLength >= s.length) {
                bestN = s;
                bestLength = s.length;
            }
        }
        bestN.computeRelationsRecursiveStep(results, dir, input, v);
    }


    private void computeRelationsRecursiveStep(List<Range> results, boolean dir, List<Range> input, long v) {
        if(v == visitedComputeRelations) return;
        visitedComputeRelations = v;

        for(Range r: dir ? parents : children) {
            if((!dir && r.containsAll(input)) || (dir && r.containedIn(input))) {
                r.optimize(results, dir, input, v);
            } else if((!dir && !r.negation.containedIn(input)) || (dir && !r.outsideOfAll(input))) {
                r.computeRelationsRecursiveStep(results, dir, input, v);
            }
        }
    }


    private void linkRelations(List<Range> pSet, List<Range> cSet, long v) {
        for(Range p: pSet) {
            addLink(p, this);
        }
        for(Range c: cSet) {
            c.visitedLinkRelations = v;
            addLink(this, c);
        }

        for(Range p: pSet) {
            ArrayList<Range> tmp = new ArrayList<>();
            for(Range c: p.children) {
                if(c.visitedLinkRelations == v) {
                    tmp.add(c);
                }
            }

            for(Range c: tmp) {
                removeLink(p, c);
            }
        }
    }


    private void optimize(List<Range> results, boolean dir, List<Range> input, long v) {
        if(v == visitedOptimize) return;
        visitedOptimize = v;

        boolean f = false;
        for(Range r: dir ? children : parents) {
            if(v != r.visitedComputeRelations) {
                if ((!dir && r.containsAll(input)) || (dir && r.containedIn(input))) {
                    f = true;
                    r.optimize(results, dir, input, v);
                }
            }
        }

        if(!f) {
            results.add(this);
        }
    }


    private void remove() {
        assert !inv;
        assert !isRemoved;
        isRemoved = true;
        removedId = removedIdCounter++;

        for(Range p: parents) {
            p.children.remove(this);
            if(p != negation) {
                p.negation.parents.remove(negation);
            }
        }
        for(Range c: children) {
            c.parents.remove(this);
            if(this != c.negation) {
                c.negation.children.remove(negation);
            }
        }
        for(Range p: parents) {
            for(Range c: children) {
                if(!c.isLinked(p, visitedCounter++)) {
                    addLink(p, c);
                }
            }
        }

        parents = null;
        children = null;
        negation.negation = null;
        negation = null;
    }


    public static void addLink(Range a, Range b) {
        a.children.add(b);
        b.parents.add(a);
        if(a != b.negation) {
            b.negation.children.add(a.negation);
            a.negation.parents.add(b.negation);
        }
    }


    public static void removeLink(Range a, Range b) {
        a.children.remove(b);
        b.parents.remove(a);
        if(a != b.negation) {
            b.negation.children.remove(a.negation);
            a.negation.parents.remove(b.negation);
        }
    }


    private boolean isLinked(Range n, long v) {
        assert visitedContains <= v;
        assert !isRemoved;
        assert !n.isRemoved;

        if(this == n) {
            return true;
        }

        visitedContains = v;
        if(length < n.length) return false;

        for(Range p: parents) {
            if(p.visitedContains != v && p.isLinked(n, v)) return true;
        }
        return false;
    }


    public boolean containedIn(Collection<Range> input) {
        if(inv) return false;
        if(containedInAny(input)) {
            return true;
        } else if(outsideOfAll(input)) {
            return false;
        }
        for(Range p: parents) {
            if(!p.containedIn(input)) return false;
        }
        return true;
    }


    public boolean containsAll(List<Range> input) {
        for(Range n: input) {
            if(!contains(n)) return false;
        }
        return true;
    }


    public boolean outsideOfAll(Collection<Range> input) {
        for(Range n: input) {
            if(!n.negation.contains(this)) return false;
        }
        return true;
    }


    public boolean containedInAny(Collection<Range> input) {
        for(Range n: input) {
            if(n.contains(this)) return true;
        }
        return false;
    }


    public boolean contains(int p) {
        for(int[] seg: segments) {
            if(seg[0] <= p && p < seg[1]) return true;
        }
        return false;
    }


    public static boolean contains(List<int[]> r, int p) {
        for(int[] seg: r) {
            if(seg[0] <= p && p < seg[1]) return true;
        }
        return false;
    }


    public boolean contains(Range r) {
        return contains(segments, r.segments);
    }


    public static boolean contains(List<int[]> ra, List<int[]> rb) {
        int i = 0;
        int j = 0;

        while(i < ra.size() && j < rb.size()) {
            int[] segA = ra.get(i);
            int[] segB = rb.get(j);
            if(segA[1] <= segB[0]) {
                i++;
            } else if(segA[0] > segB[0] || segA[1] < segB[1]) {
                return false;
            } else {
                j++;
            }
        }
        return j == rb.size();
    }


    public static boolean overlaps(Range ra, Range rb, boolean includeAdjoined, boolean onlyAfter) {
        return overlaps(ra.segments, rb.segments, includeAdjoined, onlyAfter);
    }


    public static boolean overlaps(List<int[]> ra, List<int[]> rb, boolean includeAdjoined, boolean onlyAfter) {
        if(ra.size() == 1 && rb.size() == 1) return simpleOverlaps(ra.get(0), rb.get(0), includeAdjoined, onlyAfter);

        List<int[]> intersect = intersection(includeAdjoined ? extend(ra, onlyAfter ? 0 : 1, 1) : ra, rb);
        if(!onlyAfter) {
            return intersect.size() != 0;
        } else {
            return intersect.size() != 0 && !isAnySegmentBefore(intersect, complement(ra, intersect));
        }
    }


    private static boolean simpleOverlaps(int[] sa, int[] sb, boolean includeAdjoined, boolean onlyAfter) {
        int bs = includeAdjoined && !onlyAfter ? 1 : 0;
        int es = includeAdjoined ? 1 : 0;
        return !(sa[1] + es <= sb[0] || sb[1] <= sa[0] - bs) && (!onlyAfter || sb[0] >= sa[0]);
    }


    static boolean isAnySegmentBefore(List<int[]> ra, List<int[]> rb) {
        int i = 0;
        int j = 0;

        while(i < ra.size() && j < rb.size()) {
            int[] sa = ra.get(i);
            int[] sb = rb.get(j);

            if(sa[1] == sb[0]) return true;

            int a = Integer.compare(sa[1], sb[1]);
            if(a <= 0) i++;
            if(a >= 0) j++;
        }
        return false;
    }


    public static List<int[]> extend(List<int[]> r, int before, int after) {
        ArrayList<int[]> results = new ArrayList<>();
        for(int[] s: r) {
            int begin = s[0] > before ? s[0] - before : 0;
            int end = Integer.MAX_VALUE - s[1] < after ? Integer.MAX_VALUE : s[1] + after;

            results.add(new int[] {begin, end});
        }

        return mergeAdjoined(results);
    }


    private static List<int[]> union(Range[] r) {
        List<int[]> tmp = invert(r[0].segments);
        for(int i = 1; i < r.length; i++) {
            tmp = intersection(tmp, invert(r[i].segments));
        }

        return invert(tmp);
    }


    public static List<int[]> union(List<List<int[]>> r) {
        int s = 0;
        if(s == 0) return new ArrayList<>();
        if(s == 1) return r.get(0);
        List<int[]> tmp = invert(r.get(0));
        for(int i = 1; i < r.size(); i++) {
            tmp = intersection(tmp, invert(r.get(i)));
        }

        return invert(tmp);
    }


    public static List<int[]> intersection(List<int[]> segA, List<int[]> segB) {
        int i = 0;
        int j = 0;

        ArrayList<int[]> results = new ArrayList<>();
        while(i < segA.size() && j < segB.size()) {
            int[] sa = segA.get(i);
            int[] sb = segB.get(j);

            int begin = Math.max(sa[0], sb[0]);
            int end = Math.min(sa[1], sb[1]);
            if(begin < end) {
                results.add(new int[] {begin, end});
            }

            int a = Integer.compare(sa[1], sb[1]);
            if(a <= 0) i++;
            if(a >= 0) j++;
        }

        return mergeAdjoined(results);
    }


    public Range intersection(Range r) {
        return Range.createInternal(doc, intersection(segments, r.segments));
    }


    public Range complement(Range r) {
        return intersection(r.negation);
    }


    public static List<int[]> complement(List<int[]> ra, List<int[]> rb) {
        return intersection(ra, invert(rb));
    }


    private static List<int[]> invert(List<int[]> segments) {
        List<int[]> results = new ArrayList<>();
        int pos = 0;

        for(int[] seg: segments) {
            if(pos < seg[0]) {
                results.add(new int[] {pos, seg[0]});
            }

            pos = seg[1];
        }

        if(pos < Integer.MAX_VALUE) {
            results.add(new int[] {pos, Integer.MAX_VALUE});
        }

        return mergeAdjoined(results);
    }


    public boolean isGapLess() {
        return segments.size() == 1;
    }


    public int[] getSegment(int i) {
        return segments.get(i);
    }


    public List<int[]> getSegments() {
        return segments;
    }


    public int getBegin() {
        return segments.get(0)[0];
    }


    public int getEnd() {
        return segments.get(segments.size() - 1)[1];
    }


    public int getBegin(boolean invert) {
        return invert ? getEnd() : getBegin();
    }


    public int getEnd(boolean invert) {
        return invert ? getBegin() : getEnd();
    }


    public boolean isEmpty() {
        return segments.isEmpty();
    }


    public boolean isTop() {
        return doc.topRange == this;
    }


    public boolean isBottom() {
        return doc.bottomRange == this;
    }


    public boolean contains(boolean dir, Range n) {
        boolean r;
        if(!dir) {
            r = contains(n, visitedCounter++);
        } else {
            r = n.contains(this, visitedCounter++);
        }
        return r;
    }


    private boolean contains(Range n, long v) {
        assert visitedContains <= v;
        assert !isRemoved;
        assert !n.isRemoved;

        if(!inv && n.inv) {
            return false;
        }

        /*
        wenn geprüft werden soll ob a (inv) b (!inv) enthält, kann mit b.contains(a.negation) getestet werden ob der test überhaupt sinn macht.
        */

        if(this == n || isTop() || n.isBottom()) {
            return true;
        }

        visitedContains = v;
        if(length < n.length) return false;

        if(getEnd() < n.getBegin() || n.getEnd() < getBegin()) {
            return inv != n.inv;
        }

        boolean result = false;
        for(Range p: parents) {
            if(p.visitedContains != v && p.contains(n, v)) {
                result = true;
                break;
            }
        }
        return result;
    }


    public void countRef() {
        if(isBottom() || isTop()) return;
        refCount++;
    }


    public void releaseRef() {
        if(isBottom() || isTop()) return;
        assert refCount > 0;
        refCount--;
        if(refCount == 0) {
            remove();
        }
    }


    public Collection<Activation> getOverlappingInputActivations() {
        ArrayList<Activation> results = new ArrayList<>();

        List<Range> tmp = new ArrayList<>();
        collectOverlapping(tmp, this, true, false, false, visitedCounter++);
        for(Range r: tmp) {
            results.addAll(r.inputActivations.values());
        }
        return results;
    }


    public void collect(List<Range> results, boolean dir, boolean includeFirst, long v) {
        if(inv) return;
        if(visitedCollect == v) return;
        visitedCollect = v;

        if(includeFirst && !inv) {
            results.add(this);
        }

        for(Range r: dir ? parents : children) {
            r.collect(results, dir, true, v);
        }
    }


    public void collectOverlapping(List<Range> results, Range r, boolean test, boolean includeAdjoined, boolean onlyAfter, long v) {
        visitedCollect = v;

        if(test && !overlaps(this, r, includeAdjoined, onlyAfter)) return;

        results.add(this);

        for(Range c: children) {
            if(!c.inv && c.visitedCollect != v) {
                c.collectOverlapping(results, r, false, includeAdjoined, onlyAfter, v);
            }
        }
        for(Range p: parents) {
            if(!p.inv && p.visitedCollect != v) {
                p.collectOverlapping(results, r, true, includeAdjoined, onlyAfter, v);
            }
        }
    }


    public List<Range> select(Relation rr) {
        ArrayList<Range> results = new ArrayList<>();
        switch (rr) {
            case EQUALS:
                results.add(this);
                break;
            case CONTAINS:
                collect(results, false, true, visitedCounter++);
                break;
            case CONTAINED_IN:
                collect(results, true, true, visitedCounter++);
                break;
            case OVERLAPS:
                collectOverlapping(results, this, true, false, false, Range.visitedCounter++);
                break;
            case OVERLAPS_INCLUDE_ADJOINED:
                collectOverlapping(results, this, true, true, false, Range.visitedCounter++);
                break;
            case OVERLAPS_AFTER:
                collectOverlapping(results, this, true, true, true, Range.visitedCounter++);
                break;
            default:
        }

        return results;
    }


    public String getText() {
        StringBuilder sb = new StringBuilder();
        for(int[] s: segments) {
            sb.append(doc.getContent().substring(Math.min(s[0], doc.length()), Math.min(s[1], doc.length())));
        }

        return sb.toString();
    }


    public String toString() {
        StringBuilder sb = new StringBuilder();

        if(segments.size() > 1) {
            sb.append("(");
        }

        for(int i = 0; i < segments.size(); i++) {
            if(i > 0) {
                sb.append(", ");
            }
            sb.append("(");
            sb.append(segments.get(i)[0]);
            sb.append(",");
            sb.append(segments.get(i)[1]);
            sb.append(")");
        }

        if(segments.size() > 1) {
            sb.append(")");
        }

        return sb.toString();
    }


    @Override
    public int compareTo(Range r) {
        int i = 0;

        while(i < segments.size() && i < r.segments.size()) {
            int[] sa = segments.get(i);
            int[] sb = r.segments.get(i);

            int a = Integer.compare(sa[0], sb[0]);
            if(a != 0) return a;
            int b = Integer.compare(sa[1], sb[1]);
            if(b != 0) return b;
            i++;
        }

        int c = Integer.compare(segments.size(), r.segments.size());
        return c;
    }
}
