package org.aika.network.neuron;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.aika.corpus.Option;
import org.aika.corpus.Range;
import org.aika.network.Iteration;
import org.aika.network.Model;
import org.aika.network.neuron.simple.lattice.AndNode;
import org.aika.network.neuron.simple.lattice.InputNode;
import org.aika.network.neuron.simple.lattice.NegativeInputNode;
import org.aika.network.neuron.simple.lattice.PositiveInputNode;

import java.util.*;


/**
 *
 * @author Lukas Molzberger
 */
public class Neuron implements Comparable<Neuron> {

    public Model m;

    public static final int MAX_RECURRENT_ACTIVATIONS = 10;

    public static int currentNeuronId = 0;
    public int id = currentNeuronId++;
    public String label;

    public double bias;

    public SortedMap<Neuron, Synapse> inputSynapses = new TreeMap<>();
    public SortedMap<Neuron, Synapse> outputSynapses = new TreeMap<>();

    public SortedMap<Input.InputKey, Input> outputNodes = new TreeMap<>();

    public Node node;

    public boolean inferenceMode;

    public boolean isPredefined;

    public boolean isPublished;

    public boolean isBlocked;


    public Neuron() {}


    public Neuron(String label) {
        this.label = label;
    }


    public Neuron(String label, boolean isBlocked) {
        this.label = label;
        this.isBlocked = isBlocked;
    }


    public void publish(Model m) {
        m.publishedNeurons.add(this);

        InputNode.add(m, null, this, false);
//        InputNode.add(m, this, true);
    }


    public void unpublish() {
        m.publishedNeurons.remove(this);

        for(Input out: outputNodes.values()) {
            out.remove(m);
        }
    }


    public PositiveInputNode getPositiveOutputNode(Integer rid) {
        return (PositiveInputNode) outputNodes.get(new Input.InputKey(PositiveInputNode.INPUT_TYPE, rid));
    }


    public NegativeInputNode getNegativeOutputNode(Integer rid) {
        return (NegativeInputNode) outputNodes.get(new Input.InputKey(NegativeInputNode.INPUT_TYPE, rid));
    }


    public void addInputActivation(Iteration t, int begin, int end) {
        Range r = Range.create(t.doc, begin, end);
        Node.addActivationAndPropagate(t, false, new Activation.Key(node, r, 0, t.doc.bottom, 0), r, null, Collections.EMPTY_SET, Collections.EMPTY_SET);

        t.propagate();
    }


    public void addInputActivation(Iteration t, int begin, int end, Option o) {
        Range r = Range.create(t.doc, begin, end);
        Node.addActivationAndPropagate(t, false, new Activation.Key(node, r, 0, o, 0), r, null, Collections.EMPTY_SET, Collections.EMPTY_SET);

        t.propagate();
    }


    public void removeInputActivation(Iteration t, int begin, int end) {
        Range r = Range.create(t.doc, begin, end);
        Node.removeActivationAndPropagate(t, false, new Activation(new Activation.Key(node, r, 0, t.doc.bottom, 0)), r);

        t.propagate();
    }


    public void removeInputActivation(Iteration t, int begin, int end, Option o) {
        Range r = Range.create(t.doc, begin, end);
        Node.removeActivationAndPropagate(t, false, new Activation(new Activation.Key(node, r, 0, o, 0)), r);

        t.propagate();
    }




    public void propagateAddedActivation(Iteration t, Activation act, Range addedRange) {
        for(Input out: outputNodes.values()) {
            out.addActivation(t, act, addedRange);
        }
    }


    public void propagateRemovedActivation(Iteration t, Activation act, Range removedRange) {
        for(Input out: outputNodes.values()) {
            out.removeActivation(t, act, removedRange);
        }
    }


    public void propagateInputFrequencyChange(Iteration t) {
        for(Input i: outputNodes.subMap(new Input.InputKey(PositiveInputNode.INPUT_TYPE, Integer.MIN_VALUE), new Input.InputKey(NegativeInputNode.INPUT_TYPE, Integer.MAX_VALUE)).values()) {
            InputNode in = (InputNode) i;
            if(m.trainingInterval.contains(in.frequency)) {
                for (AndNode n : in.getAndChildPatterns(new HashSet<>())) {
                    t.weightChanged.add(n);
                }
            }
        }
    }


    public void remove() {
        unpublish();

        for(Synapse is: inputSynapses.values()) {
            is.input.outputSynapses.remove(is.output);
        }
        for(Synapse is: outputSynapses.values()) {
            is.output.inputSynapses.remove(is.input);
        }
    }


    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("N(");
        sb.append(id);
        if(label != null) {
            sb.append(",");
            sb.append(label);
        }
        sb.append(")");
        return sb.toString();
    }


    public String toStringWithSynapses() {
        SortedSet<Synapse> is = new TreeSet<>(new Comparator<Synapse>() {
            @Override
            public int compare(Synapse s1, Synapse s2) {
                int r = Float.compare(s2.w, s1.w);
                if(r != 0) return r;
                return Integer.compare(s1.input.id, s2.input.id);
            }
        });

        StringBuilder sb = new StringBuilder();
        sb.append(toString());
        sb.append("<");
        sb.append("B:");
        sb.append(bias);
        sb.append(", ");
        for(Synapse s: is) {
            sb.append(s.w);
            sb.append(":");
            sb.append(s.input.toString());
        }
        sb.append(">");
        return sb.toString();
    }


    @Override
    public int compareTo(Neuron n) {
        if(id < n.id) return -1;
        else if(id > n.id) return 1;
        else return 0;
    }

}
