package org.aika.network.neuron.recurrent;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


import org.aika.network.Model;
import org.aika.network.neuron.Input;
import org.aika.network.neuron.Neuron;
import org.aika.network.neuron.Synapse;


/**
 *
 * @author Lukas Molzberger
 */
public class RecurrentNeuron extends Neuron {

    public static final int MAX_RECURRENT_ACTIVATIONS = 10;


    public RecurrentNeuron() {}


    public RecurrentNeuron(String label) {
        this.label = label;
    }


    public RecurrentNeuron(String label, boolean isBlocked) {
        this.label = label;
        this.isBlocked = isBlocked;
    }


    public static RecurrentNeuron create(Model m, RecurrentNeuron n, Synapse inputSignal, Synapse clockSignal, Synapse terminationSignal, boolean direction, int maxLength, boolean predefined) {
        n.m = m;
        n.isPredefined = predefined;

        InputNode inputNode = null;
        if(inputSignal != null) {
            n.inputSynapses.put(inputSignal.input, inputSignal);
            Input.InputKey ik = new Input.InputKey(InputNode.INPUT_TYPE, null);
            if(inputSignal.input.outputNodes.get(ik) == null) {
                inputNode = new InputNode(m);
                inputNode.inputNeuron = inputSignal.input;
                inputSignal.input.outputNodes.put(ik, inputNode);
            } else inputNode = (InputNode) inputSignal.input.outputNodes.get(ik);
        }

        ClockNode clockNode = null;
        if(clockSignal != null) {
            n.inputSynapses.put(clockSignal.input, clockSignal);
            Input.InputKey ik = new Input.InputKey(ClockNode.INPUT_TYPE, null);
            if(clockSignal.input.outputNodes.get(ik) == null) {
                clockNode = new ClockNode(m);
                clockNode.inputNeuron = clockSignal.input;
                clockSignal.input.outputNodes.put(ik, clockNode);
            } else clockNode = (ClockNode) clockSignal.input.outputNodes.get(ik);
        }

        TerminationNode terminationNode = null;
        if(terminationSignal != null) {
            n.inputSynapses.put(terminationSignal.input, terminationSignal);
            Input.InputKey ik = new Input.InputKey(ClockNode.INPUT_TYPE, null);
            if(terminationSignal.input.outputNodes.get(ik) == null) {
                terminationNode = new TerminationNode(m);
                terminationNode.inputNeuron = terminationSignal.input;
                terminationSignal.input.outputNodes.put(ik, terminationNode);
            } else terminationNode = (TerminationNode) terminationSignal.input.outputNodes.get(ik);
        }


        ClockTerminationNode ctNode = clockNode.ctNodes.get(terminationNode);
        if(ctNode == null) {
            ctNode = new ClockTerminationNode(m, direction);
            clockNode.ctNodes.put(terminationNode, ctNode);
            ctNode.clockNode = clockNode;
            if(terminationNode != null) {
                terminationNode.ctNodes.put(clockNode, ctNode);
                ctNode.terminationNode = terminationNode;
            }
            m.clockTerminationNodes.add(ctNode);
        }

        OutputNode outputNode = ctNode.outputNodes.get(inputNode);
        if(outputNode == null) {
            outputNode = new OutputNode(m, maxLength);
            ctNode.outputNodes.put(inputNode, outputNode);
            inputNode.outputNodes.put(ctNode, outputNode);

            outputNode.ctNode = ctNode;
            outputNode.inputNode = inputNode;
        }

        outputNode.neuron = n;
        n.node = outputNode;

        n.publish(m);

        return n;
    }


    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("RN(");
        sb.append(id);
        if(label != null) {
            sb.append(",");
            sb.append(label);
        }
        sb.append(")");
        return sb.toString();
    }
}
