package org.aika.network.neuron.recurrent;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


import org.aika.corpus.Option;
import org.aika.corpus.Range;
import org.aika.network.Iteration;
import org.aika.network.Model;
import org.aika.network.neuron.Activation;
import org.aika.network.neuron.Neuron;
import org.aika.network.neuron.Node;

import java.util.Collection;
import java.util.Collections;


/**
 *
 * @author Lukas Molzberger
 */
public abstract class RecurrentNode extends Node {

    public boolean direction;
    public int frequency;


    public enum RecurrentType {
        INPUT_SIGNAL,
        CLOCK_SIGNAL,
        TERMINATION_SIGNAL,
    }


    public RecurrentNode(Model m, int level) {
        super(m, level);
    }


    private Activation.Key computeActivationKey(Iteration t, Activation inputAct) {
        Option o = inputAct.newOption != null ? Option.add(t.doc, true, inputAct.key.o, inputAct.newOption) : inputAct.key.o;

        if(o == null) return null;

        return new Activation.Key(this, inputAct.key.pos, 0, o, inputAct.key.fired);
    }


    public void addActivation(Iteration t, Activation inputAct, Range addedRange) {
        Activation.Key ak = computeActivationKey(t, inputAct);

        if(ak != null) {
            addActivationAndPropagate(t, false, ak, addedRange, null, Collections.singleton(inputAct), Collections.singleton(inputAct));
        }
    }


    public void removeActivation(Iteration t, Activation inputAct, Range removedRange) {
        Activation.Key ak = computeActivationKey(t, inputAct);

        if(ak != null) {
            removeActivationAndPropagate(t, false, Activation.get(this, ak), removedRange);
        }
    }


    public abstract Collection<? extends RecurrentNode> getChildren();


    public void propagateAddedActivation(Iteration t, Activation act, Range addedRange, Option conflict) {
        if(conflict != null) return;

        for(RecurrentNode rn: getChildren()) {
            rn.addActivation(t, act, addedRange);
        }
    }


    public void propagateRemovedActivation(Iteration t, Activation act, Range removedRange) {
        for(RecurrentNode rn: getChildren()) {
            rn.removeActivation(t, act, removedRange);
        }
    }


    @Override
    public void initActivation(Iteration t, Activation act) {
    }


    @Override
    public void deleteActivation(Iteration t, Activation act) {
    }


    @Override
    public boolean isAllowedOption(Option n, Activation act, long v) {
        return false;
    }


    @Override
    public boolean isNegative() {
        return false;
    }


    @Override
    public boolean containsNegative() {
        return false;
    }


    @Override
    public void cleanup(Model m) {

    }


    @Override
    public double computeSynapseWeightSum(Neuron n) {
        return 0;
    }
}
