package org.aika.network.neuron.simple;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.aika.network.Model;
import org.aika.network.neuron.Neuron;
import org.aika.network.neuron.Synapse;
import org.aika.network.neuron.recurrent.RecurrentNeuron;
import org.aika.network.neuron.simple.lattice.LatticeNode;

import java.util.Set;


/**
 *
 * @author Lukas Molzberger
 */
public class SimpleNeuron extends Neuron {


    public SimpleNeuron() {}


    public SimpleNeuron(String label) {
        this.label = label;
    }


    public SimpleNeuron(String label, boolean isBlocked) {
        super(label, isBlocked);
    }



    public void remove() {
        unpublish();

        for(Synapse is: inputSynapses.values()) {
            is.input.outputSynapses.remove(is.output);
        }
        for(Synapse is: outputSynapses.values()) {
            is.output.inputSynapses.remove(is.input);
        }
    }


    public static SimpleNeuron create(Model m, SimpleNeuron n, double bias, Set<Synapse> inputs, boolean predefined, boolean inferenceMode) {
        n.m = m;
        n.isPredefined = predefined;
        for(Synapse s: inputs) {
            n.inputSynapses.put(s.input, s);
            s.output = n;
        }
        n.bias = bias;

        n.inferenceMode = inferenceMode;
        n = LatticeNode.addNeuron(n, inputs);

        n.publish(m);
        return n;
    }


    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("SN(");
        sb.append(id);
        if(label != null) {
            sb.append(",");
            sb.append(label);
        }
        sb.append(")");
        return sb.toString();
    }



    public static class Input implements Comparable<Input> {

        public final boolean isNeg;
        public final boolean isOptional;
        public final Neuron inputNeuron;

        public final int rid;
        public final boolean relative;

        public Input(boolean isNeg, SimpleNeuron inputNeuron) {
            this.isNeg = isNeg;
            this.isOptional = false;
            this.inputNeuron = inputNeuron;

            rid = 0;
            relative = true;
        }

        public Input(boolean isNeg, boolean isOptional, SimpleNeuron inputNeuron) {
            this.isNeg = isNeg;
            this.isOptional = isOptional;
            this.inputNeuron = inputNeuron;

            rid = 0;
            relative = true;
        }

        public Input(boolean isNeg, RecurrentNeuron inputNeuron, int rid, boolean relative) {
            this.isNeg = isNeg;
            this.isOptional = false;
            this.inputNeuron = inputNeuron;
            this.rid = rid;
            this.relative = relative;
        }

        public Input(boolean isNeg, boolean isOptional, RecurrentNeuron inputNeuron, int rid, boolean relative) {
            this.isNeg = isNeg;
            this.isOptional = isOptional;
            this.inputNeuron = inputNeuron;
            this.rid = rid;
            this.relative = relative;
        }

        @Override
        public int compareTo(Input in) {
            int r = Boolean.compare(isNeg, in.isNeg);
            if(r != 0) return r;
            r = Boolean.compare(isOptional, in.isOptional);
            if(r != 0) return r;
            r = inputNeuron.compareTo(in.inputNeuron);
            if(r != 0) return r;
            r = Integer.compare(rid, in.rid);
            if(r != 0) return r;
            return Boolean.compare(relative, in.relative);
        }
    }
}
