package org.aika.network.neuron.simple.lattice;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.aika.corpus.Conflicts;
import org.aika.corpus.Document;
import org.aika.corpus.Option;
import org.aika.corpus.Range;
import org.aika.network.Iteration;
import org.aika.network.Model;
import org.aika.network.neuron.Activation;
import org.aika.network.neuron.Activation.Key;

import java.util.ArrayList;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 *
 * @author Lukas Molzberger
 */
public class NegativeInputNode extends InputNode {

    public final static int INPUT_TYPE = 1;


    public NegativeInputNode(Model m, Integer rid) {
        super(m, rid);
    }


    public static TreeMap<Range, Set<Conflicts.Key>> getNegationSegments(Document doc, Set<NegativeInputNode> nn, Range r) {
        TreeSet<Integer> positions = new TreeSet<>();

        for(int[] seg: r.getSegments()) {
            positions.add(seg[0]);
            positions.add(seg[1]);
        }

        for(NegativeInputNode n: nn) {
            for(Activation act: Activation.select(n, null, r, Range.Relation.OVERLAPS, null, null, null, null, null, true)) {
                int begin = act.key.pos.getBegin();
                if(r.contains(begin)) positions.add(begin);
                int end = act.key.pos.getEnd();
                if(r.contains(end)) positions.add(end);
            }
        }

        TreeMap<Range, Set<Conflicts.Key>> results = new TreeMap<>();
        Integer lastP = null;
        for(Integer p: positions) {
            if(lastP != null) {
                Range nr = Range.create(doc, lastP, p);
                if(r.contains(nr)) {
                    Set<Conflicts.Key> opts = new TreeSet<>();
                    for(NegativeInputNode n: nn) {
                        for(Activation act : Activation.select(n, null, nr, Range.Relation.OVERLAPS, null, null, null, null, null, true)) {
                            opts.add(new Conflicts.Key(act.key.o, n));
                        }
                    }
                    results.put(nr, opts);
                }
            }
            lastP = p;
        }
        return results;
    }


    @Override
    public boolean isNegative() {
        return true;
    }


    @Override
    public boolean containsNegative() {
        return true;
    }


    @Override
    public double computeForwardWeight(Activation act) {
        return 1.0;
    }


    public double getNodeWeight(Activation act) {
        return weight;
    }



    public void propagateAddedActivation(Iteration t, Activation act, Range addedRange, Option conflict) {
        if(conflict != null) return;

        ArrayList<LatticeNode> publicNodes = new ArrayList<>();
        collectPublicNodes(publicNodes, visitedCounter++);

        for(LatticeNode n: publicNodes) {
            for(Activation nAct: Activation.select(n, null, addedRange, Range.Relation.OVERLAPS, null, null, null, null, null, true)) {
                if (!n.isAllowedOption(act.key.o, nAct, visitedCounter++)) {
                    if (addedRange.contains(nAct.key.pos)) {
                        assert nAct.initialOption != null;

                        Conflicts.add(t, this, nAct.initialOption, act.key.o);
                    } else {
                        n.removeActivationAndPropagate(t, true, nAct, nAct.key.pos.intersection(addedRange));

                        for (int[] seg : nAct.key.pos.intersection(addedRange).getSegments()) {
                            Range r = Range.create(t.doc, seg);
                            Option nio = nAct.initialOption.clonePrimitive(t);
                            Conflicts.add(t, this, nio, act.key.o);
                            n.addActivationAndPropagate(t, true, new Key(n, r, act.key.rid, nAct.key.o, nAct.key.fired), r, nio, nAct.inputs, nAct.directInputs);
                        }
                    }
                }
            }
        }
    }


    public void propagateRemovedActivation(Iteration t, Activation act, Range removedRange) {
        ArrayList<LatticeNode> publicNodes = new ArrayList<>();
        collectPublicNodes(publicNodes, visitedCounter++);

        for(LatticeNode n: publicNodes) {
            for(Activation nAct: Activation.select(n, null, removedRange, Range.Relation.OVERLAPS, null, null, null, null, null, true)) {
                if (!n.isAllowedOption(act.key.o, nAct, visitedCounter++)) {
                    if (removedRange.contains(nAct.key.pos)) {
                        assert nAct.initialOption != null;

                        Conflicts.remove(t, this, nAct.initialOption, act.key.o);
                    } else {
                        n.removeActivationAndPropagate(t, true, nAct, nAct.key.pos.intersection(removedRange));

                        for (int[] seg : nAct.key.pos.intersection(removedRange).getSegments()) {
                            Range r = Range.create(t.doc, seg);
                            Option nio = nAct.initialOption.clonePrimitive(t);
                            Conflicts.remove(t, this, nio, act.key.o);
                            n.addActivationAndPropagate(t, true, new Key(n, r, act.key.rid, nAct.key.o, nAct.key.fired), r, nio, nAct.inputs, nAct.directInputs);
                        }
                    }
                }
            }
        }
    }


    @Override
    public int getSign() {
        return -1;
    }


    @Override
    public void cleanup(Model m) {
    }


    @Override
    public void remove(Model m) {
        inputNeuron.outputNodes.remove(new InputKey(INPUT_TYPE, rid));
        super.remove(m);
    }
}
