package org.aika.network.neuron.simple.lattice;

/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.aika.corpus.Option;
import org.aika.corpus.Range;
import org.aika.network.Iteration;
import org.aika.network.Model;
import org.aika.network.neuron.Activation;
import org.aika.network.neuron.Activation.Key;
import org.aika.network.neuron.Neuron;
import org.aika.network.neuron.Node;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;


/**
 *
 * @author Lukas Molzberger
 */
public class OrNode extends LogicNode {

    public Map<LatticeNode, Double> parents = new TreeMap<>();


    public OrNode(Model m, int level) {
        super(m, level);
    }


    @Override
    public boolean isNegative() {
        return false;
    }


    @Override
    public boolean isAllowedOption(Option n, Activation act, long v) {
        return false;
    }


    @Override
    public boolean containsNegative() {
        return false;
    }


    @Override
    public void initActivation(Iteration t, Activation act) {
    }


    @Override
    public void deleteActivation(Iteration t, Activation act) {
    }


    @Override
    public double computeForwardWeight(Activation act) {
        assert !act.inputs.isEmpty();
        double maxForwardsWeight = act.computeAverageInputWeight();

        // TODO: shadowed acts might only partially overlap the current act.
        for(Activation shdAct : act.shadows.values()) {
            assert !shdAct.inputs.isEmpty();
            maxForwardsWeight = Math.max(maxForwardsWeight, shdAct.computeAverageInputWeight());
        }
        return maxForwardsWeight;
    }


    public double getNodeWeight(Activation act) {
//        assert act.directInputs.size() == 1;

        Double iw = parents.get(act.directInputs.first().key.n);
        return weight * (iw != null ? iw : 1.0);
    }


    public void addActivation(Iteration t, Key ak, Range addedRange, Activation inputAct) {
        addActivationAndPropagate(t, true, ak, addedRange, null, inputAct.inputs, Collections.singleton(inputAct));
    }


    protected void removeActivation(Iteration t, Activation act, Range removedRange) {
        removeActivationAndPropagate(t, true, act, removedRange);
    }


    @Override
    public double computeSynapseWeightSum(Neuron n) {
        throw new UnsupportedOperationException();
    }


    @Override
    public void cleanup(Model m) {

    }


    @Override
    public void expandToNextLevel(Iteration t, Activation act, Range addedRange, Option conflict, boolean train) {
        OrNode.processCandidate(t, this, act, addedRange, conflict, train);
    }


    public static void processCandidate(Iteration t, LatticeNode parentNode, Activation act, Range addedRange, Option conflict, boolean train) {
        Key ak = act.key;
        for(OrNode n: parentNode.orChildren) {
            n.addActivation(t, new Key(n, ak.pos, 0, act.newOption != null ? Option.add(t.doc, false, ak.o, act.newOption) : ak.o, ak.fired, parentNode.id), addedRange, act);
        }
    }


    @Override
    protected void collectNodeAndRefinements(AndNode.Refinement newRef, Set<AndNode.Refinement> inputs) {
        throw new UnsupportedOperationException();
    }


    public void addInput(LatticeNode inputNode) {
        inputNode.orChildren.add(this);
        parents.put(inputNode, 1.0);
    }


    public void setWeight(LatticeNode inputNode, double weight) {
        if(parents.containsKey(inputNode)) {
            parents.put(inputNode, weight);
        }
    }


    public void remove(Model m) {
        super.remove(m);

        for(LatticeNode n: parents.keySet()) {
            n.orChildren.remove(this);
        }
    }


    public String logicToString() {
        StringBuilder sb = new StringBuilder();
        sb.append("OR[");
        boolean first = true;
        int i = 0;
        for(Node n: parents.keySet()) {
            if (!first) {
                sb.append(",");
            }
            first = false;
            sb.append(n.logicToString());
            if(i > 10) {
                sb.append(",...");
                break;
            }

            i++;
        }
        sb.append("]");
        return sb.toString();
    }
}
