/*
 * Copyright 2011 Andrew Oberstar
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ajoberstar.gradle.pmd

import org.gradle.api.file.FileCollection
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.SourceTask
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.VerificationTask

/**
 * <p>
 * Gradle task that runs a PMD analysis on your code.    
 * </p>
 * <p>
 * This implementation calls the PMD Ant task.
 * </p>
 * <p>
 * See {link: http://pmd.sourceforge.net/} for more information
 * about the tool.
 * </p>
 * @author Andrew Oberstar
 * @version 0.1.0
 * @since 0.1.0
 */
class PMD extends SourceTask implements VerificationTask {	
	/**
	 * The rulesets to use during this execution.
	 */
	@Input Set<File> rulesets
	/**
	 * Whether or not to allow the build to continue if there are warnings.
	 */
	boolean ignoreFailures
	
	/**
	 * The location to generate the XML results to.
	 */
	@OutputFile File resultsFile
	/**
	 * The location to generate the HTML report to.
	 */
	@OutputFile File reportsFile
	
	/**
	 * Run the PMD analysis.
	 * 
	 * <ul>
	 * <li>{@code source} the source files to analyze</li>
	 * <li>{@code rulesets} the rules to evaluate against the code</li>
	 * <li>{@code resultsFile} the location to generate the XML</li>
	 * <li>(@code reportsFile} the location to generate the HTML</li>
	 * </ul>
	 */
	@TaskAction
	void check() {
		ant.taskdef(name:'pmd', classname:'net.sourceforge.pmd.ant.PMDTask', classpath:project.configurations[PMDPlugin.PMD_CONFIGURATION_NAME].asPath)
		ant.pmd(failOnRuleViolation:!ignoreFailures) {
			source.addToAntBuilder(ant, 'fileset', FileCollection.AntType.FileSet)
			rulesets.each {
				ruleset(it)
			}
			formatter(type:'html', toFile:getReportsFile())
			formatter(type:'xml', toFile:getResultsFile())
		}
	}
	
	/**
	* Sets whether warnings generated by Findbugs will
	* stop the build.
	* @param ignoreFailures
	* @return {@code this}
	*/
	@Override
	VerificationTask setIgnoreFailures(boolean ignoreFailures) {
		this.ignoreFailures = ignoreFailures
		return this
	}
}
