/*
 * Decompiled with CFR 0.152.
 */
package alluxio.examples;

import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.InstancedConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.examples.RelatedS3Operations;
import alluxio.underfs.UfsDirectoryStatus;
import alluxio.underfs.UfsFileStatus;
import alluxio.underfs.UfsStatus;
import alluxio.underfs.UnderFileSystem;
import alluxio.underfs.options.CreateOptions;
import alluxio.underfs.options.DeleteOptions;
import alluxio.underfs.options.ListOptions;
import alluxio.underfs.options.MkdirsOptions;
import alluxio.underfs.options.OpenOptions;
import alluxio.util.CommonUtils;
import alluxio.util.UnderFileSystemUtils;
import alluxio.util.WaitForOptions;
import alluxio.util.io.PathUtils;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class UnderFileSystemCommonOperations {
    private static final Logger LOG = LoggerFactory.getLogger(UnderFileSystemCommonOperations.class);
    private static final byte[] TEST_BYTES = "TestBytes".getBytes();
    private static final String FILE_CONTENT_LENGTH_INCORRECT = "The content length of the written file is %s but expected %s";
    private static final String FILE_CONTENT_INCORRECT = "The content of the written file is incorrect";
    private static final String FILE_EXISTS_CHECK_SHOULD_SUCCEED = "Should succeed in UnderFileSystem.exists() check, but failed";
    private static final String FILE_EXISTS_CHECK_SHOULD_FAILED = "Should failed in UnderFileSystem.exists() check, but succeed";
    private static final String FILE_STATUS_RESULT_INCORRECT = "The result of UnderFileSystem.getFileStatus() is incorrect";
    private static final String IS_FAIL_CHECK_SHOULD_SUCCEED = "Should succeed in UnderFileSystem.isFile() check, but failed";
    private static final String IS_FAIL_CHECK_SHOULD_FAILED = "Should failed in UnderFileSystem.isFile() check, but succeed";
    private static final String IS_DIRECTORY_CHECK_SHOULD_SUCCEED = "Should succeed in UnderFileSystem.isDirectory() check, but failed";
    private static final String IS_DIRECTORY_CHECK_SHOULD_FAILED = "Should failed in UnderFileSystem.isDirectory() check, but succeed";
    private static final String LIST_STATUS_RESULT_INCORRECT = "The result of UnderFileSystem.listStatus() is incorrect";
    private final InstancedConfiguration mConfiguration;
    private final UnderFileSystem mUfs;
    private final String mUfsPath;
    private final String mTopLevelTestDirectory;

    public UnderFileSystemCommonOperations(String ufsPath, String topLevelTestDirectory, UnderFileSystem ufs, InstancedConfiguration configuration) {
        this.mUfsPath = ufsPath;
        this.mTopLevelTestDirectory = topLevelTestDirectory;
        this.mUfs = ufs;
        this.mConfiguration = configuration;
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void createAtomicTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createAtomic");
        OutputStream stream = this.mUfs.create(testFile, CreateOptions.defaults((AlluxioConfiguration)this.mConfiguration).setEnsureAtomic(true));
        stream.write(TEST_BYTES);
        if (this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        stream.close();
        if (!this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void createEmptyTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createEmpty");
        this.createEmptyFile(testFile);
        if (!this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={})
    public void createNoParentTest() throws IOException {
        if (!UnderFileSystemUtils.isLocal((UnderFileSystem)this.mUfs)) {
            return;
        }
        boolean haveIOException = false;
        try {
            String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createNoParent/testFile");
            OutputStream o = this.mUfs.create(testFile, CreateOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
            o.close();
        }
        catch (IOException e) {
            haveIOException = true;
        }
        if (!haveIOException) {
            throw new IOException("Expected to have IOException but do not have");
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void createParentTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createParent/testFile");
        OutputStream o = this.mUfs.create(testFile, CreateOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(true));
        o.close();
        if (!this.mUfs.exists(testFile)) {
            throw new IOException(FILE_EXISTS_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"upload", "getObject"})
    public void createOpenTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createOpen");
        this.createTestBytesFile(testFile);
        byte[] buf = new byte[TEST_BYTES.length];
        int bytesRead = this.mUfs.open(testFile).read(buf);
        if (TEST_BYTES.length != bytesRead || !Arrays.equals(buf, TEST_BYTES)) {
            throw new IOException(FILE_CONTENT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"upload", "getObject"})
    public void createOpenEmptyTest() throws IOException {
        boolean bytesReadCorrect;
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createOpenEmpty");
        this.createEmptyFile(testFile);
        byte[] buf = new byte[]{};
        int bytesRead = this.mUfs.open(testFile).read(buf);
        boolean bl = bytesReadCorrect = bytesRead == 0;
        if (UnderFileSystemUtils.isHdfs((UnderFileSystem)this.mUfs) && bytesRead == -1) {
            bytesReadCorrect = true;
        }
        if (!bytesReadCorrect) {
            throw new IOException(String.format(FILE_CONTENT_LENGTH_INCORRECT, bytesRead, 0));
        }
    }

    @RelatedS3Operations(operations={"upload", "getObject"})
    public void createOpenAtPositionTest() throws IOException {
        int[] offsets;
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createOpenAtPosition");
        this.prepareMultiBlockFile(testFile);
        for (int offset : offsets = new int[]{0, 256, 511, 512, 513, 768, 1024, 1025}) {
            InputStream inputStream = this.mUfs.open(testFile, OpenOptions.defaults().setOffset((long)offset));
            if (TEST_BYTES[offset % TEST_BYTES.length] != inputStream.read()) {
                throw new IOException(FILE_CONTENT_INCORRECT);
            }
            inputStream.close();
        }
    }

    @RelatedS3Operations(operations={"upload", "getObject"})
    public void createOpenLargeTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createOpenLarge");
        int numCopies = this.prepareMultiBlockFile(testFile);
        InputStream inputStream = this.mUfs.open(testFile);
        byte[] buf = new byte[numCopies * TEST_BYTES.length];
        int offset = 0;
        int noReadCount = 0;
        while (offset < buf.length && noReadCount < 3) {
            int bytesRead;
            try {
                bytesRead = inputStream.read(buf, offset, buf.length - offset);
            }
            catch (Exception e) {
                LOG.info("Failed to read from file {}: {}", (Object)testFile, (Object)e.toString());
                bytesRead = -1;
            }
            if (bytesRead != -1) {
                noReadCount = 0;
                for (int i = 0; i < bytesRead; ++i) {
                    if (TEST_BYTES[(offset + i) % TEST_BYTES.length] == buf[offset + i]) continue;
                    throw new IOException(FILE_CONTENT_INCORRECT);
                }
                offset += bytesRead;
                continue;
            }
            ++noReadCount;
        }
        if (noReadCount > 3) {
            throw new IOException("Too many retries in reading the written large file");
        }
    }

    @RelatedS3Operations(operations={"upload", "getObject"})
    public void createOpenExistingLargeFileTest() throws IOException {
        int offset;
        int bytesRead;
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"createOpenExistingLargeFile");
        int numCopies = this.prepareMultiBlockFile(testFile);
        InputStream inputStream = this.mUfs.openExistingFile(testFile);
        byte[] buf = new byte[numCopies * TEST_BYTES.length];
        for (offset = 0; offset < buf.length && (bytesRead = inputStream.read(buf, offset, buf.length - offset)) != -1; offset += bytesRead) {
            for (int i = 0; i < bytesRead; ++i) {
                if (TEST_BYTES[(offset + i) % TEST_BYTES.length] == buf[offset + i]) continue;
                throw new IOException(FILE_CONTENT_INCORRECT);
            }
        }
        if (buf.length != offset) {
            throw new IOException(FILE_CONTENT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"upload", "deleteObject", "getObjectMetadata"})
    public void deleteFileTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"deleteFile");
        this.createEmptyFile(testFile);
        this.mUfs.deleteFile(testFile);
        if (this.mUfs.exists(testFile)) {
            throw new IOException(FILE_EXISTS_CHECK_SHOULD_FAILED);
        }
        if (this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "deleteObjects", "listObjectsV2", "getObjectMetadata"})
    public void deleteDirTest() throws IOException {
        String testDirEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"deleteDirTestDirEmpty");
        String testDirNonEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"deleteDirTestDirNonEmpty1");
        String testDirNonEmptyChildDir = PathUtils.concatPath((Object)testDirNonEmpty, (Object)"deleteDirTestDirNonEmpty2");
        String testDirNonEmptyChildFile = PathUtils.concatPath((Object)testDirNonEmpty, (Object)"deleteDirTestDirNonEmptyF");
        String testDirNonEmptyChildDirFile = PathUtils.concatPath((Object)testDirNonEmptyChildDir, (Object)"deleteDirTestDirNonEmptyChildDirF");
        this.mUfs.mkdirs(testDirEmpty, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(testDirNonEmpty, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(testDirNonEmptyChildDir, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(testDirNonEmptyChildFile);
        this.createEmptyFile(testDirNonEmptyChildDirFile);
        this.mUfs.deleteDirectory(testDirEmpty, DeleteOptions.defaults().setRecursive(false));
        if (this.mUfs.isDirectory(testDirEmpty)) {
            throw new IOException("Directory is deleted but succeed in UnderFileSystem.isDirectory() check");
        }
        try {
            this.mUfs.deleteDirectory(testDirNonEmpty, DeleteOptions.defaults().setRecursive(false));
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (!this.mUfs.isDirectory(testDirNonEmpty)) {
            throw new IOException("Created directory should succeed in UnderFileSystem.isDirectory() check, but failed");
        }
        this.mUfs.deleteDirectory(testDirNonEmpty, DeleteOptions.defaults().setRecursive(true));
        if (this.mUfs.isDirectory(testDirNonEmpty) || this.mUfs.isDirectory(testDirNonEmptyChildDir) || this.mUfs.isFile(testDirNonEmptyChildFile) || this.mUfs.isFile(testDirNonEmptyChildDirFile)) {
            throw new IOException("Deleted file or directory still exist");
        }
    }

    @RelatedS3Operations(operations={"putObject", "deleteObjects", "listObjectsV2", "getObjectMetadata"})
    public void deleteLargeDirectoryTest() throws IOException {
        String[] children;
        LargeDirectoryConfig config = this.prepareLargeDirectory();
        this.mUfs.deleteDirectory(config.getTopLevelDirectory(), DeleteOptions.defaults().setRecursive(true));
        for (String child : children = config.getChildren()) {
            boolean childDeleted = false;
            for (int i = 0; i < 20; ++i) {
                boolean bl = childDeleted = !this.mUfs.isFile(child) && !this.mUfs.isDirectory(child);
                if (childDeleted) break;
                CommonUtils.sleepMs((long)500L);
            }
            if (childDeleted) continue;
            throw new IOException("Deleted file or directory still exist");
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata", "deleteObject"})
    public void createDeleteFileConjuctionTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"deleteThenCreateNonexistingFile");
        this.createTestBytesFile(testFile);
        if (!this.mUfs.exists(testFile)) {
            throw new IOException(FILE_EXISTS_CHECK_SHOULD_SUCCEED);
        }
        if (!this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
        this.mUfs.deleteExistingFile(testFile);
        if (this.mUfs.exists(testFile)) {
            throw new IOException(FILE_EXISTS_CHECK_SHOULD_FAILED);
        }
        OutputStream o = this.mUfs.createNonexistingFile(testFile);
        o.write(TEST_BYTES);
        o.close();
        if (!this.mUfs.exists(testFile)) {
            throw new IOException(FILE_EXISTS_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "deleteObjects", "listObjectsV2", "getObjectMetadata"})
    public void createThenDeleteExistingDirectoryTest() throws IOException {
        LargeDirectoryConfig config = this.prepareLargeDirectory();
        if (!this.mUfs.deleteExistingDirectory(config.getTopLevelDirectory(), DeleteOptions.defaults().setRecursive(true))) {
            throw new IOException("Failed to delete existing directory");
        }
    }

    @RelatedS3Operations(operations={"putObject", "getObjectMetadata"})
    public void existsTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFile");
        if (this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        this.createEmptyFile(testFile);
        if (!this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDir");
        if (this.mUfs.isDirectory(testDir)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_FAILED);
        }
        this.mUfs.mkdirs(testDir, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        if (!this.mUfs.isDirectory(testDir)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "getObjectMetadata"})
    public void getDirectoryStatusTest() throws IOException {
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDir");
        this.mUfs.mkdirs(testDir);
        UfsStatus status = this.mUfs.getStatus(testDir);
        if (!(status instanceof UfsDirectoryStatus)) {
            throw new IOException("Failed to get ufs directory status");
        }
    }

    @RelatedS3Operations(operations={"putObject", "getObjectMetadata"})
    public void createThenGetExistingDirectoryStatusTest() throws IOException {
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDir");
        this.mUfs.mkdirs(testDir);
        UfsStatus status = this.mUfs.getExistingStatus(testDir);
        if (!(status instanceof UfsDirectoryStatus)) {
            throw new IOException("Failed to get ufs directory status");
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void getFileSizeTest() throws IOException {
        String testFileEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFileEmpty");
        String testFileNonEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFileNonEmpty");
        this.createEmptyFile(testFileEmpty);
        this.createTestBytesFile(testFileNonEmpty);
        if (this.mUfs.getFileStatus(testFileEmpty).getContentLength() != 0L || this.mUfs.getFileStatus(testFileNonEmpty).getContentLength() != (long)TEST_BYTES.length) {
            throw new IOException(FILE_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void createThenGetExistingFileStatusTest() throws IOException {
        String testFileNonEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFileNonEmpty");
        String testFileLarge = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFileLarge");
        this.createTestBytesFile(testFileNonEmpty);
        int numCopies = this.prepareMultiBlockFile(testFileLarge);
        if ((long)TEST_BYTES.length != this.mUfs.getExistingFileStatus(testFileNonEmpty).getContentLength() || (long)(TEST_BYTES.length * numCopies) != this.mUfs.getExistingFileStatus(testFileLarge).getContentLength()) {
            throw new IOException(FILE_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void getFileStatusTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFile");
        this.createEmptyFile(testFile);
        UfsStatus status = this.mUfs.getStatus(testFile);
        if (!(status instanceof UfsFileStatus)) {
            throw new IOException("Failed to get ufs file status");
        }
    }

    @RelatedS3Operations(operations={})
    public void createThenGetExistingStatusTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFile");
        this.createTestBytesFile(testFile);
        UfsStatus status = this.mUfs.getExistingStatus(testFile);
        if (!(status instanceof UfsFileStatus)) {
            throw new IOException("Failed to get ufs file status");
        }
    }

    @RelatedS3Operations(operations={"upload", "getObjectMetadata"})
    public void getModTimeTest() throws IOException {
        long slack = 5000L;
        long start = System.currentTimeMillis();
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFile");
        this.createTestBytesFile(testFile);
        long end = System.currentTimeMillis();
        long modTime = this.mUfs.getFileStatus(testFile).getLastModifiedTime();
        if (modTime < start - slack || modTime > end + slack) {
            throw new IOException(FILE_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"getObjectMetadata"})
    public void getNonExistingDirectoryStatusTest() throws IOException {
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"nonExistentDir");
        try {
            this.mUfs.getDirectoryStatus(testDir);
        }
        catch (FileNotFoundException e) {
            return;
        }
        throw new IOException("Get status on a non-existent directory did not through " + FileNotFoundException.class);
    }

    @RelatedS3Operations(operations={"getObjectMetadata"})
    public void getNonExistingFileStatusTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"nonExistentFile");
        try {
            this.mUfs.getFileStatus(testFile);
        }
        catch (FileNotFoundException e) {
            return;
        }
        throw new IOException("Get file status on a non-existent file did not through " + FileNotFoundException.class);
    }

    @RelatedS3Operations(operations={"getObjectMetadata"})
    public void getNonExistingPathStatusTest() throws IOException {
        String testPath = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"nonExistentPath");
        try {
            this.mUfs.getStatus(testPath);
        }
        catch (FileNotFoundException e) {
            return;
        }
        throw new IOException("Get status on a non-existent path did not through " + FileNotFoundException.class);
    }

    @RelatedS3Operations(operations={"putObject", "deleteObject"})
    public void isFileTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testFile");
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDir");
        if (this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        this.createEmptyFile(testFile);
        this.mUfs.mkdirs(testDir, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        if (!this.mUfs.isFile(testFile)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
        if (this.mUfs.isFile(testDir)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void listStatusTest() throws IOException {
        String testDirNonEmpty = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDirNonEmpty1");
        String testDirNonEmptyChildDir = PathUtils.concatPath((Object)testDirNonEmpty, (Object)"testDirNonEmpty2");
        String testDirNonEmptyChildFile = PathUtils.concatPath((Object)testDirNonEmpty, (Object)"testDirNonEmptyF");
        String testDirNonEmptyChildDirFile = PathUtils.concatPath((Object)testDirNonEmptyChildDir, (Object)"testDirNonEmptyChildDirF");
        this.mUfs.mkdirs(testDirNonEmpty, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(testDirNonEmptyChildDir, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(testDirNonEmptyChildFile);
        this.createEmptyFile(testDirNonEmptyChildDirFile);
        Object[] expectedResTopDir = new String[]{"testDirNonEmpty2", "testDirNonEmptyF"};
        Object[] expectedResTopDir2 = new String[]{"/testDirNonEmpty2", "/testDirNonEmptyF"};
        Arrays.sort(expectedResTopDir);
        Arrays.sort(expectedResTopDir2);
        UfsStatus[] resTopDirStatus = this.mUfs.listStatus(testDirNonEmpty);
        Object[] resTopDir = UfsStatus.convertToNames((UfsStatus[])resTopDirStatus);
        Arrays.sort(resTopDir);
        if (!Arrays.equals(expectedResTopDir, resTopDir) && !Arrays.equals(expectedResTopDir2, resTopDir)) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        if (!this.mUfs.listStatus(testDirNonEmptyChildDir)[0].getName().equals("testDirNonEmptyChildDirF") || this.mUfs.listStatus(testDirNonEmptyChildDir)[0].getName().equals("/testDirNonEmptyChildDirF")) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (int i = 0; i < resTopDir.length; ++i) {
            if (this.mUfs.isDirectory(PathUtils.concatPath((Object)testDirNonEmpty, (Object)resTopDirStatus[i].getName())) == resTopDirStatus[i].isDirectory()) continue;
            throw new IOException("UnderFileSystem.isDirectory() result is different from expected");
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void listStatusEmptyTest() throws IOException {
        String testDir = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"listStatusEmpty");
        this.mUfs.mkdirs(testDir);
        UfsStatus[] res = this.mUfs.listStatus(testDir);
        if (res.length != 0) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void listStatusFileTest() throws IOException {
        String testFile = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"listStatusFile");
        this.createEmptyFile(testFile);
        if (this.mUfs.listStatus(testFile) != null) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void listLargeDirectoryTest() throws IOException {
        LargeDirectoryConfig config = this.prepareLargeDirectory();
        String[] children = config.getChildren();
        UfsStatus[] results = new UfsStatus[]{};
        for (int i = 0; i < 50 && children.length != (results = this.mUfs.listStatus(config.getTopLevelDirectory())).length; ++i) {
            CommonUtils.sleepMs((long)500L);
        }
        if (children.length != results.length) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        Object[] resultNames = UfsStatus.convertToNames((UfsStatus[])results);
        Arrays.sort(resultNames);
        for (int i = 0; i < children.length; ++i) {
            if (((String)resultNames[i]).equals(CommonUtils.stripPrefixIfPresent((String)children[i], (String)PathUtils.normalizePath((String)config.getTopLevelDirectory(), (String)"/")))) continue;
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void listStatusRecursiveTest() throws IOException {
        String root = this.mTopLevelTestDirectory;
        this.mUfs.mkdirs(root);
        if (this.mUfs.listStatus(root).length != 0) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        String sub1 = PathUtils.concatPath((Object)root, (Object)"sub1");
        String sub2 = PathUtils.concatPath((Object)root, (Object)"sub2");
        String sub11 = PathUtils.concatPath((Object)sub1, (Object)"sub11");
        String file11 = PathUtils.concatPath((Object)sub11, (Object)"file11");
        String file2 = PathUtils.concatPath((Object)sub2, (Object)"file2");
        String file = PathUtils.concatPath((Object)root, (Object)"file");
        if (this.mUfs.listStatus(sub1, ListOptions.defaults().setRecursive(true)) != null) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        this.mUfs.mkdirs(sub1, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(sub2, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(sub11, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(file11);
        this.createEmptyFile(file2);
        this.createEmptyFile(file);
        Object[] expectedResRoot = new String[]{"sub1", "sub2", "sub1/sub11", "sub1/sub11/file11", "sub2/file2", "file"};
        UfsStatus[] actualResRootStatus = this.mUfs.listStatus(root, ListOptions.defaults().setRecursive(true));
        Object[] actualResRoot = UfsStatus.convertToNames((UfsStatus[])actualResRootStatus);
        Arrays.sort(expectedResRoot);
        Arrays.sort(actualResRoot);
        if (!Arrays.equals(expectedResRoot, actualResRoot)) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (int i = 0; i < actualResRoot.length; ++i) {
            if (this.mUfs.isDirectory(PathUtils.concatPath((Object)root, (Object)actualResRootStatus[i].getName())) == actualResRootStatus[i].isDirectory()) continue;
            throw new IOException("UnderFileSystem.isDirectory() result is different from expected");
        }
        Object[] expectedResSub1 = new String[]{"sub11", "sub11/file11"};
        Object[] actualResSub1 = UfsStatus.convertToNames((UfsStatus[])this.mUfs.listStatus(sub1, ListOptions.defaults().setRecursive(true)));
        Arrays.sort(expectedResSub1);
        Arrays.sort(actualResSub1);
        if (!Arrays.equals(expectedResSub1, actualResSub1) || this.mUfs.listStatus(file, ListOptions.defaults().setRecursive(true)) != null) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void mkdirsTest() throws IOException {
        this.mUfs.mkdirs(this.mTopLevelTestDirectory);
        if (this.mUfs.listStatus(this.mTopLevelTestDirectory).length != 0) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        String testDirTop = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"testDirTop");
        String testDir1 = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"1");
        String testDir2 = PathUtils.concatPath((Object)testDir1, (Object)"2");
        String testDir3 = PathUtils.concatPath((Object)testDir2, (Object)"3");
        String testDirDeep = PathUtils.concatPath((Object)testDir3, (Object)"testDirDeep");
        this.mUfs.mkdirs(testDirTop, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.mUfs.mkdirs(testDirDeep, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(true));
        if (!(this.mUfs.isDirectory(testDirTop) && this.mUfs.isDirectory(testDir1) && this.mUfs.isDirectory(testDir2) && this.mUfs.isDirectory(testDir3) && this.mUfs.isDirectory(testDirDeep))) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void objectCommonPrefixesIsDirectoryTest() throws IOException {
        if (!this.mUfs.isObjectStorage()) {
            return;
        }
        ObjectStorePreConfig config = this.prepareObjectStore();
        String baseDirectoryPath = config.getBaseDirectoryPath();
        if (!this.mUfs.isDirectory(baseDirectoryPath)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
        for (String subDirName : config.getSubDirectoryNames()) {
            String subDirPath = PathUtils.concatPath((Object)baseDirectoryPath, (Object)subDirName);
            if (this.mUfs.isDirectory(subDirPath)) continue;
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void objectCommonPrefixesListStatusNonRecursiveTest() throws IOException {
        int i;
        int foundIndex;
        if (!this.mUfs.isObjectStorage()) {
            return;
        }
        ObjectStorePreConfig config = this.prepareObjectStore();
        String baseDirectoryPath = config.getBaseDirectoryPath();
        UfsStatus[] results = this.mUfs.listStatus(baseDirectoryPath);
        if (config.getSubDirectoryNames().length + config.getFileNames().length != results.length) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (String fileName : config.getFileNames()) {
            foundIndex = -1;
            for (i = 0; i < results.length; ++i) {
                if (!results[i].getName().equals(fileName)) continue;
                foundIndex = i;
            }
            if (foundIndex >= 0 && results[foundIndex].isFile()) continue;
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (String subDirName : config.getSubDirectoryNames()) {
            foundIndex = -1;
            for (i = 0; i < results.length; ++i) {
                if (!results[i].getName().equals(subDirName)) continue;
                foundIndex = i;
            }
            if (foundIndex >= 0 && results[foundIndex].isDirectory()) continue;
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void objectCommonPrefixesListStatusRecursiveTest() throws IOException {
        if (!this.mUfs.isObjectStorage()) {
            return;
        }
        ObjectStorePreConfig config = this.prepareObjectStore();
        String baseDirectoryPath = config.getBaseDirectoryPath();
        UfsStatus[] results = this.mUfs.listStatus(baseDirectoryPath, ListOptions.defaults().setRecursive(true));
        String[] fileNames = config.getFileNames();
        String[] subDirNames = config.getSubDirectoryNames();
        if (subDirNames.length + fileNames.length + subDirNames.length * fileNames.length != results.length) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (String fileName : fileNames) {
            int foundIndex = -1;
            for (int i = 0; i < results.length; ++i) {
                if (!results[i].getName().equals(fileName)) continue;
                foundIndex = i;
            }
            if (foundIndex >= 0 && results[foundIndex].isFile()) continue;
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        for (String subDirName : subDirNames) {
            int dirIndex = -1;
            for (int i = 0; i < results.length; ++i) {
                if (!results[i].getName().equals(subDirName)) continue;
                dirIndex = i;
            }
            if (dirIndex < 0 || !results[dirIndex].isDirectory()) {
                throw new IOException(LIST_STATUS_RESULT_INCORRECT);
            }
            for (String fileName : config.getFileNames()) {
                int fileIndex = -1;
                for (int i = 0; i < results.length; ++i) {
                    if (!results[i].getName().equals(String.format("%s/%s", subDirName, fileName))) continue;
                    fileIndex = i;
                }
                if (fileIndex >= 0 && results[fileIndex].isFile()) continue;
                throw new IOException(LIST_STATUS_RESULT_INCORRECT);
            }
        }
    }

    @RelatedS3Operations(operations={"putObject", "listObjectsV2", "getObjectMetadata"})
    public void objectNestedDirsListStatusRecursiveTest() throws IOException {
        if (!this.mUfs.isObjectStorage()) {
            return;
        }
        String root = this.mTopLevelTestDirectory;
        int nesting = 5;
        String path = root;
        for (int i = 0; i < nesting; ++i) {
            path = PathUtils.concatPath((Object)path, (Object)("dir" + i));
        }
        String file1 = PathUtils.concatPath((Object)path, (Object)"file.txt");
        if (this.mUfs.listStatus(root, ListOptions.defaults().setRecursive(true)).length != 0) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
        String fileKey = file1.substring(PathUtils.normalizePath((String)this.mUfsPath, (String)"/").length());
        if (!this.mUfs.mkdirs(fileKey)) {
            throw new IOException("Failed to create empty object");
        }
        path = "";
        ArrayList<String> paths = new ArrayList<String>();
        for (int i = 0; i < nesting; ++i) {
            path = i == 0 ? "dir" + i : PathUtils.concatPath((Object)path, (Object)("dir" + i));
            paths.add(path);
        }
        path = PathUtils.concatPath((Object)path, (Object)"file.txt");
        paths.add(path);
        Object[] expectedStatus = paths.toArray(new String[paths.size()]);
        Object[] actualStatus = UfsStatus.convertToNames((UfsStatus[])this.mUfs.listStatus(root, ListOptions.defaults().setRecursive(true)));
        Arrays.sort(expectedStatus);
        Arrays.sort(actualStatus);
        if (expectedStatus.length != actualStatus.length || !Arrays.equals(expectedStatus, actualStatus)) {
            throw new IOException(LIST_STATUS_RESULT_INCORRECT);
        }
    }

    @RelatedS3Operations(operations={"upload", "copyObject", "deleteObject", "getObjectMetadata"})
    public void renameFileTest() throws IOException {
        String testFileSrc = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameFileSrc");
        String testFileDst = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameFileDst");
        this.createEmptyFile(testFileSrc);
        this.mUfs.renameFile(testFileSrc, testFileDst);
        if (this.mUfs.isFile(testFileSrc)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        if (!this.mUfs.isFile(testFileDst)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"upload", "copyObject", "deleteObject", "getObjectMetadata"})
    public void renameRenamableFileTest() throws IOException {
        String testFileSrc = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameFileSrc");
        String testFileDst = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameFileDst");
        this.prepareMultiBlockFile(testFileSrc);
        this.mUfs.renameRenamableFile(testFileSrc, testFileDst);
        if (this.mUfs.isFile(testFileSrc)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        if (!this.mUfs.isFile(testFileDst)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "upload", "copyObject", "listObjectsV2", "getObjectMetadata"})
    public void renameDirectoryTest() throws IOException {
        String testDirSrc = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameDirectorySrc");
        String testDirSrcChild = PathUtils.concatPath((Object)testDirSrc, (Object)"testFile");
        String testDirDst = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameDirectoryDst");
        String testDirDstChild = PathUtils.concatPath((Object)testDirDst, (Object)"testFile");
        this.mUfs.mkdirs(testDirSrc, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(testDirSrcChild);
        this.mUfs.renameDirectory(testDirSrc, testDirDst);
        if (this.mUfs.isDirectory(testDirSrc)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_FAILED);
        }
        if (this.mUfs.isFile(testDirSrcChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        if (!this.mUfs.isDirectory(testDirDst)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
        if (!this.mUfs.isFile(testDirDstChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "upload", "copyObject", "listObjectsV2", "getObjectMetadata"})
    public void renameDirectoryDeepTest() throws IOException {
        String testDirSrc = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameDirectoryDeepSrc");
        String testDirSrcChild = PathUtils.concatPath((Object)testDirSrc, (Object)"testFile");
        String testDirSrcNested = PathUtils.concatPath((Object)testDirSrc, (Object)"testNested");
        String testDirSrcNestedChild = PathUtils.concatPath((Object)testDirSrcNested, (Object)"testNestedFile");
        String testDirDst = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameDirectoryDeepDst");
        String testDirDstChild = PathUtils.concatPath((Object)testDirDst, (Object)"testFile");
        String testDirDstNested = PathUtils.concatPath((Object)testDirDst, (Object)"testNested");
        String testDirDstNestedChild = PathUtils.concatPath((Object)testDirDstNested, (Object)"testNestedFile");
        this.mUfs.mkdirs(testDirSrc, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(testDirSrcChild);
        this.mUfs.mkdirs(testDirSrcNested, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.createEmptyFile(testDirSrcNestedChild);
        this.mUfs.renameDirectory(testDirSrc, testDirDst);
        if (this.mUfs.isDirectory(testDirSrc) || this.mUfs.isDirectory(testDirSrcNested)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_FAILED);
        }
        if (this.mUfs.isFile(testDirSrcChild) || this.mUfs.isFile(testDirSrcNestedChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        if (!this.mUfs.isDirectory(testDirDst) || !this.mUfs.isDirectory(testDirDstNested)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
        if (!this.mUfs.isFile(testDirDstChild) || !this.mUfs.isFile(testDirDstNestedChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "upload", "copyObject", "listObjectsV2", "getObjectMetadata"})
    public void renameRenamableDirectoryTest() throws IOException {
        String testDirSrc = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameRenamableDirectorySrc");
        String testDirSrcChild = PathUtils.concatPath((Object)testDirSrc, (Object)"testFile");
        String testDirSrcNested = PathUtils.concatPath((Object)testDirSrc, (Object)"testNested");
        String testDirSrcNestedChild = PathUtils.concatPath((Object)testDirSrcNested, (Object)"testNestedFile");
        String testDirDst = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"renameRenamableDirectoryDst");
        String testDirDstChild = PathUtils.concatPath((Object)testDirDst, (Object)"testFile");
        String testDirDstNested = PathUtils.concatPath((Object)testDirDst, (Object)"testNested");
        String testDirDstNestedChild = PathUtils.concatPath((Object)testDirDstNested, (Object)"testNestedFile");
        this.mUfs.mkdirs(testDirSrc, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.prepareMultiBlockFile(testDirSrcChild);
        this.mUfs.mkdirs(testDirSrcNested, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        this.prepareMultiBlockFile(testDirSrcNestedChild);
        this.mUfs.renameRenamableDirectory(testDirSrc, testDirDst);
        if (this.mUfs.isDirectory(testDirSrc) || this.mUfs.isDirectory(testDirSrcNested)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_FAILED);
        }
        if (this.mUfs.isFile(testDirSrcChild) || this.mUfs.isFile(testDirSrcNestedChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_FAILED);
        }
        if (!this.mUfs.isDirectory(testDirDst) || !this.mUfs.isDirectory(testDirDstNested)) {
            throw new IOException(IS_DIRECTORY_CHECK_SHOULD_SUCCEED);
        }
        if (!this.mUfs.isFile(testDirDstChild) || !this.mUfs.isFile(testDirDstNestedChild)) {
            throw new IOException(IS_FAIL_CHECK_SHOULD_SUCCEED);
        }
    }

    @RelatedS3Operations(operations={"putObject", "upload", "copyObject", "listObjectsV2", "getObjectMetadata"})
    public void renameLargeDirectoryTest() throws Exception {
        String[] srcChildren;
        LargeDirectoryConfig config = this.prepareLargeDirectory();
        String dstTopLevelDirectory = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"topLevelDirMoved");
        this.mUfs.renameDirectory(config.getTopLevelDirectory(), dstTopLevelDirectory);
        for (String src : srcChildren = config.getChildren()) {
            CommonUtils.waitFor((String)"list after delete consistency", () -> {
                try {
                    return !this.mUfs.exists(src);
                }
                catch (IOException e) {
                    return false;
                }
            }, (WaitForOptions)WaitForOptions.defaults().setTimeoutMs(10000).setInterval(500));
        }
        CommonUtils.waitFor((String)"list after create consistency", () -> {
            try {
                UfsStatus[] results = this.mUfs.listStatus(dstTopLevelDirectory);
                if (srcChildren.length != results.length) {
                    return false;
                }
                Object[] resultNames = UfsStatus.convertToNames((UfsStatus[])results);
                Arrays.sort(resultNames);
                for (int i = 0; i < srcChildren.length; ++i) {
                    if (((String)resultNames[i]).equals(CommonUtils.stripPrefixIfPresent((String)srcChildren[i], (String)PathUtils.normalizePath((String)config.getTopLevelDirectory(), (String)"/")))) continue;
                    throw new IOException("Destination is different from source after rename directory");
                }
            }
            catch (IOException e) {
                return false;
            }
            return true;
        }, (WaitForOptions)WaitForOptions.defaults().setTimeoutMs(10000).setInterval(500));
    }

    private void createEmptyFile(String path) throws IOException {
        OutputStream o = this.mUfs.create(path);
        o.close();
    }

    private void createTestBytesFile(String path) throws IOException {
        OutputStream o = this.mUfs.create(path);
        o.write(TEST_BYTES);
        o.close();
    }

    private LargeDirectoryConfig prepareLargeDirectory() throws IOException {
        int i;
        String filePrefix = "a_";
        String folderPrefix = "b_";
        String topLevelDirectory = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)"topLevelDir");
        int numFiles = 100;
        String[] children = new String[200];
        this.mUfs.mkdirs(topLevelDirectory, MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        for (i = 0; i < 100; ++i) {
            children[i] = PathUtils.concatPath((Object)topLevelDirectory, (Object)("a_" + String.format("%04d", i)));
            this.createEmptyFile(children[i]);
        }
        for (i = 0; i < 100; ++i) {
            children[100 + i] = PathUtils.concatPath((Object)topLevelDirectory, (Object)("b_" + String.format("%04d", i)));
            this.mUfs.mkdirs(children[100 + i], MkdirsOptions.defaults((AlluxioConfiguration)this.mConfiguration).setCreateParent(false));
        }
        return new LargeDirectoryConfig(topLevelDirectory, children);
    }

    private int prepareMultiBlockFile(String testFile) throws IOException {
        OutputStream outputStream = this.mUfs.create(testFile);
        int numBlocks = 3;
        int blockSize = (int)this.mConfiguration.getBytes(PropertyKey.USER_BLOCK_SIZE_BYTES_DEFAULT);
        int numCopies = numBlocks * blockSize / TEST_BYTES.length;
        for (int i = 0; i < numCopies; ++i) {
            outputStream.write(TEST_BYTES);
        }
        outputStream.close();
        return numCopies;
    }

    private ObjectStorePreConfig prepareObjectStore() throws IOException {
        String[] childrenFiles;
        String baseDirectoryName = "base";
        String baseDirectoryPath = PathUtils.concatPath((Object)this.mTopLevelTestDirectory, (Object)baseDirectoryName);
        String baseDirectoryKey = baseDirectoryPath.substring(PathUtils.normalizePath((String)this.mUfsPath, (String)"/").length());
        String[] subDirectories = new String[]{"a", "b", "c"};
        for (String child : childrenFiles = new String[]{"sample1.jpg", "sample2.jpg", "sample3.jpg"}) {
            this.mUfs.create(String.format("%s/%s", baseDirectoryKey, child)).close();
        }
        for (String subdir : subDirectories) {
            for (String child : childrenFiles) {
                this.mUfs.create(String.format("%s/%s/%s", baseDirectoryKey, subdir, child)).close();
            }
        }
        return new ObjectStorePreConfig(baseDirectoryPath, childrenFiles, subDirectories);
    }

    private class ObjectStorePreConfig {
        private String mBaseDirectoryPath;
        private String[] mSubDirectoryNames;
        private String[] mFileNames;

        ObjectStorePreConfig(String baseDirectoryKey, String[] childrenFiles, String[] subDirectories) {
            this.mBaseDirectoryPath = baseDirectoryKey;
            this.mFileNames = childrenFiles;
            this.mSubDirectoryNames = subDirectories;
        }

        String getBaseDirectoryPath() {
            return this.mBaseDirectoryPath;
        }

        String[] getFileNames() {
            return this.mFileNames;
        }

        String[] getSubDirectoryNames() {
            return this.mSubDirectoryNames;
        }
    }

    private class LargeDirectoryConfig {
        private String mTopLevelDirectory;
        private String[] mChildren;

        LargeDirectoryConfig(String topLevelDirectory, String[] children) {
            this.mTopLevelDirectory = topLevelDirectory;
            this.mChildren = children;
        }

        String getTopLevelDirectory() {
            return this.mTopLevelDirectory;
        }

        String[] getChildren() {
            return this.mChildren;
        }
    }
}

