/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.cli;

import alluxio.exception.AlluxioException;
import alluxio.exception.status.InvalidArgumentException;

import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.CommandLine;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.CommandLineParser;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.DefaultParser;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.Options;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.ParseException;

import java.alluxio.shaded.client.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * An interface for all the alluxio.shaded.client.com.ands that can be run from a shell.
 */
public interface Command {

  /**
   * Gets the alluxio.shaded.client.com.and name as input from the shell.
   *
   * @return the alluxio.shaded.client.com.and name
   */
  String getCommandName();

  /**
   * @return the supported {@link Options} of the alluxio.shaded.client.com.and
   */
  default Options getOptions() {
    return new Options();
  }

  /**
   * If a alluxio.shaded.client.com.and has sub-alluxio.shaded.client.com.ands, the first argument should be the sub-alluxio.shaded.client.com.and's name,
   * all arguments and options will be parsed for the sub-alluxio.shaded.client.com.and.
   *
   * @return whether this alluxio.shaded.client.com.and has sub-alluxio.shaded.client.com.ands
   */
  default boolean hasSubCommand() {
    return false;
  }

  /**
   * @return a map from sub-alluxio.shaded.client.com.and names to sub-alluxio.shaded.client.com.and instances
   */
  default Map<String, Command> getSubCommands() {
    return new HashMap<>();
  }

  /**
   * Parses and validates the arguments.
   *
   * @param args the arguments for the alluxio.shaded.client.com.and, excluding the alluxio.shaded.client.com.and name
   * @return the parsed alluxio.shaded.client.com.and line object
   * @throws InvalidArgumentException when arguments are not valid
   */
  default CommandLine parseAndValidateArgs(String... args) throws InvalidArgumentException {
    CommandLine cmdline;
    Options opts = getOptions();
    CommandLineParser parser = new DefaultParser();
    try {
      cmdline = parser.parse(opts, args);
    } catch (ParseException e) {
      throw new InvalidArgumentException(
          String.format("Failed to parse args for %s: %s", getCommandName(), e.getMessage()), e);
    }
    validateArgs(cmdline);
    return cmdline;
  }

  /**
   * Checks if the arguments are valid or throw InvalidArgumentException.
   *
   * @param cl the parsed alluxio.shaded.client.com.and line for the arguments
   * @throws InvalidArgumentException when arguments are not valid
   */
  default void validateArgs(CommandLine cl) throws InvalidArgumentException {}

  /**
   * Runs the alluxio.shaded.client.com.and.
   *
   * @param cl the parsed alluxio.shaded.client.com.and line for the arguments
   * @return the result of running the alluxio.shaded.client.com.and
   */
  default int run(CommandLine cl) throws AlluxioException, IOException {
    return 0;
  }

  /**
   * @return the usage information of the alluxio.shaded.client.com.and
   */
  String getUsage();

  /**
   * @return the description information of the alluxio.shaded.client.com.and
   */
  String getDescription();
}
