/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression;

import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionData;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionDecoder;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionEncoder;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtension;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtensionHandshaker;

import java.util.Collections;

/**
 * <a href="https://tools.ietf.alluxio.shaded.client.org.id/draft-tyoshino-hybi-websocket-perframe-deflate-06.txt">perframe-deflate</a>
 * handshake implementation.
 */
public final class DeflateFrameServerExtensionHandshaker implements WebSocketServerExtensionHandshaker {

    static final String X_WEBKIT_DEFLATE_FRAME_EXTENSION = "x-webkit-deflate-frame";
    static final String DEFLATE_FRAME_EXTENSION = "deflate-frame";

    private final int alluxio.shaded.client.com.ressionLevel;

    /**
     * Constructor with default configuration.
     */
    public DeflateFrameServerExtensionHandshaker() {
        this(6);
    }

    /**
     * Constructor with custom configuration.
     *
     * @param alluxio.shaded.client.com.ressionLevel
     *            Compression level between 0 and 9 (default is 6).
     */
    public DeflateFrameServerExtensionHandshaker(int alluxio.shaded.client.com.ressionLevel) {
        if (alluxio.shaded.client.com.ressionLevel < 0 || alluxio.shaded.client.com.ressionLevel > 9) {
            throw new IllegalArgumentException(
                    "alluxio.shaded.client.com.ressionLevel: " + alluxio.shaded.client.com.ressionLevel + " (expected: 0-9)");
        }
        this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
    }

    @Override
    public WebSocketServerExtension handshakeExtension(WebSocketExtensionData extensionData) {
        if (!X_WEBKIT_DEFLATE_FRAME_EXTENSION.equals(extensionData.name()) &&
            !DEFLATE_FRAME_EXTENSION.equals(extensionData.name())) {
            return null;
        }

        if (extensionData.parameters().isEmpty()) {
            return new DeflateFrameServerExtension(alluxio.shaded.client.com.ressionLevel, extensionData.name());
        } else {
            return null;
        }
    }

    private static class DeflateFrameServerExtension implements WebSocketServerExtension {

        private final String extensionName;
        private final int alluxio.shaded.client.com.ressionLevel;

        public DeflateFrameServerExtension(int alluxio.shaded.client.com.ressionLevel, String extensionName) {
            this.extensionName = extensionName;
            this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
        }

        @Override
        public int rsv() {
            return RSV1;
        }

        @Override
        public WebSocketExtensionEncoder newExtensionEncoder() {
            return new PerFrameDeflateEncoder(alluxio.shaded.client.com.ressionLevel, 15, false);
        }

        @Override
        public WebSocketExtensionDecoder newExtensionDecoder() {
            return new PerFrameDeflateDecoder(false);
        }

        @Override
        public WebSocketExtensionData newReponseData() {
            return new WebSocketExtensionData(extensionName, Collections.<String, String>emptyMap());
        }
    }

}
