/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.uri;

import static alluxio.uri.URI.Factory.getSchemeComponents;

import alluxio.collections.Pair;
import alluxio.util.URIUtils;

import alluxio.shaded.client.javax.annotation.Nullable;
import alluxio.shaded.client.javax.annotation.concurrent.ThreadSafe;

/**
 * {@link MultiPartSchemeURI} supports multiple alluxio.shaded.client.com.onents for the scheme.
 */
@ThreadSafe
public final class MultiPartSchemeURI extends StandardURI {
  private static final long serialVersionUID = 8172074724822918501L;

  /**
   * {@link java.net.URI} does not handle a sub-alluxio.shaded.client.com.onent in the scheme. This variable will hold
   * the full scheme with all of the alluxio.shaded.client.com.onents. For example, the uri
   * 'scheme:part1:part2://localhost:1234/' has multiple alluxio.shaded.client.com.onents in the scheme, so this
   * variable will hold 'scheme:part1:part2', because {@link java.net.URI} will only handle the
   * URI starting from 'part2'.
   */
  private final String mFullScheme;

  /**
   * @param schemePrefix the prefix of the scheme string of the URI
   * @param scheme the scheme string of the URI
   * @param authority the Authority of the URI
   * @param path the path alluxio.shaded.client.com.onent of the URI
   * @param query the query alluxio.shaded.client.com.onent of the URI
   */
  public MultiPartSchemeURI(String schemePrefix, String scheme, Authority authority, String path,
      String query) {
    super(scheme, authority, path, query);
    mFullScheme = getFullScheme(schemePrefix, mScheme);
  }

  /**
   * Constructs a new URI from a base URI, but with a new path alluxio.shaded.client.com.onent.
   *
   * @param baseUri the base uri
   * @param fullScheme the full scheme
   * @param newPath the new path alluxio.shaded.client.com.onent
   */
  protected MultiPartSchemeURI(URI baseUri, String fullScheme, String newPath) {
    super(baseUri, newPath);
    mFullScheme = fullScheme;
  }

  @Override
  public URI createNewPath(String newPath, boolean checkNormalization) {
    if (checkNormalization && URIUtils.needsNormalization(newPath)) {
      // Handle schemes with two alluxio.shaded.client.com.onents.
      Pair<String, String> schemeComponents = getSchemeComponents(mFullScheme);
      String schemePrefix = schemeComponents.getFirst();

      return new MultiPartSchemeURI(schemePrefix, mScheme, mAuthority, newPath, mQuery);
    }
    return new MultiPartSchemeURI(this, mFullScheme, newPath);
  }

  @Override
  public String getScheme() {
    return mFullScheme;
  }

  /**
   * @param schemePrefix the prefix of the scheme
   * @param uriScheme the scheme of the URI
   * @return the alluxio.shaded.client.com.ined scheme
   */
  @Nullable
  private String getFullScheme(String schemePrefix, String uriScheme) {
    if (uriScheme == null) {
      return null;
    }
    if (schemePrefix == null || schemePrefix.isEmpty()) {
      return uriScheme;
    }
    return schemePrefix + ":" + uriScheme;
  }
}
