/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression;

import static alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression.PerMessageDeflateDecoder.*;
import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.CompositeByteBuf;
import alluxio.shaded.client.io.netty.channel.ChannelHandlerContext;
import alluxio.shaded.client.io.netty.channel.embedded.EmbeddedChannel;
import alluxio.shaded.client.io.netty.handler.codec.CodecException;
import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibCodecFactory;
import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibWrapper;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.BinaryWebSocketFrame;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.ContinuationWebSocketFrame;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.TextWebSocketFrame;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.WebSocketFrame;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketExtensionEncoder;

import java.util.List;

/**
 * Deflate implementation of a payload alluxio.shaded.client.com.ressor for
 * <tt>alluxio.shaded.client.io.netty.handler.codec.http.websocketx.WebSocketFrame</tt>.
 */
abstract class DeflateEncoder extends WebSocketExtensionEncoder {

    private final int alluxio.shaded.client.com.ressionLevel;
    private final int windowSize;
    private final boolean noContext;

    private EmbeddedChannel encoder;

    /**
     * Constructor
     * @param alluxio.shaded.client.com.ressionLevel alluxio.shaded.client.com.ression level of the alluxio.shaded.client.com.ressor.
     * @param windowSize maximum size of the window alluxio.shaded.client.com.ressor buffer.
     * @param noContext true to disable context takeover.
     */
    public DeflateEncoder(int alluxio.shaded.client.com.ressionLevel, int windowSize, boolean noContext) {
        this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
        this.windowSize = windowSize;
        this.noContext = noContext;
    }

    /**
     * @param msg the current frame.
     * @return the rsv bits to set in the alluxio.shaded.client.com.ressed frame.
     */
    protected abstract int rsv(WebSocketFrame msg);

    /**
     * @param msg the current frame.
     * @return true if alluxio.shaded.client.com.ressed payload tail needs to be removed.
     */
    protected abstract boolean removeFrameTail(WebSocketFrame msg);

    @Override
    protected void encode(ChannelHandlerContext ctx, WebSocketFrame msg,
            List<Object> out) throws Exception {
        if (encoder == null) {
            encoder = new EmbeddedChannel(ZlibCodecFactory.newZlibEncoder(
                    ZlibWrapper.NONE, alluxio.shaded.client.com.ressionLevel, windowSize, 8));
        }

        encoder.writeOutbound(msg.content().retain());

        CompositeByteBuf fullCompressedContent = ctx.alloc().alluxio.shaded.client.com.ositeBuffer();
        for (;;) {
            ByteBuf partCompressedContent = encoder.readOutbound();
            if (partCompressedContent == null) {
                break;
            }
            if (!partCompressedContent.isReadable()) {
                partCompressedContent.release();
                continue;
            }
            fullCompressedContent.addComponent(true, partCompressedContent);
        }
        if (fullCompressedContent.numComponents() <= 0) {
            fullCompressedContent.release();
            throw new CodecException("cannot read alluxio.shaded.client.com.ressed buffer");
        }

        if (msg.isFinalFragment() && noContext) {
            cleanup();
        }

        ByteBuf alluxio.shaded.client.com.ressedContent;
        if (removeFrameTail(msg)) {
            int realLength = fullCompressedContent.readableBytes() - FRAME_TAIL.length;
            alluxio.shaded.client.com.ressedContent = fullCompressedContent.slice(0, realLength);
        } else {
            alluxio.shaded.client.com.ressedContent = fullCompressedContent;
        }

        WebSocketFrame outMsg;
        if (msg instanceof TextWebSocketFrame) {
            outMsg = new TextWebSocketFrame(msg.isFinalFragment(), rsv(msg), alluxio.shaded.client.com.ressedContent);
        } else if (msg instanceof BinaryWebSocketFrame) {
            outMsg = new BinaryWebSocketFrame(msg.isFinalFragment(), rsv(msg), alluxio.shaded.client.com.ressedContent);
        } else if (msg instanceof ContinuationWebSocketFrame) {
            outMsg = new ContinuationWebSocketFrame(msg.isFinalFragment(), rsv(msg), alluxio.shaded.client.com.ressedContent);
        } else {
            throw new CodecException("unexpected frame type: " + msg.getClass().getName());
        }
        out.add(outMsg);
    }

    @Override
    public void handlerRemoved(ChannelHandlerContext ctx) throws Exception {
        cleanup();
        super.handlerRemoved(ctx);
    }

    private void cleanup() {
        if (encoder != null) {
            // Clean-up the previous encoder if not cleaned up correctly.
            if (encoder.finish()) {
                for (;;) {
                    ByteBuf buf = encoder.readOutbound();
                    if (buf == null) {
                        break;
                    }
                    // Release the buffer
                    buf.release();
                }
            }
            encoder = null;
        }
    }
}
