/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License, version 2.0 (the
 * "License"); you may not use this file except in alluxio.shaded.client.com.liance with the License. You may obtain a
 * copy of the License at:
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.http2;

import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.Unpooled;
import alluxio.shaded.client.io.netty.channel.ChannelFuture;
import alluxio.shaded.client.io.netty.channel.ChannelHandlerContext;
import alluxio.shaded.client.io.netty.channel.ChannelPromise;
import alluxio.shaded.client.io.netty.channel.embedded.EmbeddedChannel;
import alluxio.shaded.client.io.netty.handler.codec.ByteToMessageDecoder;
import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibCodecFactory;
import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibWrapper;
import alluxio.shaded.client.io.netty.util.concurrent.PromiseCombiner;
import alluxio.shaded.client.io.netty.util.internal.UnstableApi;

import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_ENCODING;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_LENGTH;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.DEFLATE;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.GZIP;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.IDENTITY;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.X_DEFLATE;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.X_GZIP;

/**
 * A decorating HTTP2 encoder that will alluxio.shaded.client.com.ress data frames according to the {@code content-encoding} header for each
 * stream. The alluxio.shaded.client.com.ression provided by this class will be applied to the data for the entire stream.
 */
@UnstableApi
public class CompressorHttp2ConnectionEncoder extends DecoratingHttp2ConnectionEncoder {
    public static final int DEFAULT_COMPRESSION_LEVEL = 6;
    public static final int DEFAULT_WINDOW_BITS = 15;
    public static final int DEFAULT_MEM_LEVEL = 8;

    private final int alluxio.shaded.client.com.ressionLevel;
    private final int windowBits;
    private final int memLevel;
    private final Http2Connection.PropertyKey propertyKey;

    public CompressorHttp2ConnectionEncoder(Http2ConnectionEncoder delegate) {
        this(delegate, DEFAULT_COMPRESSION_LEVEL, DEFAULT_WINDOW_BITS, DEFAULT_MEM_LEVEL);
    }

    public CompressorHttp2ConnectionEncoder(Http2ConnectionEncoder delegate, int alluxio.shaded.client.com.ressionLevel, int windowBits,
                                            int memLevel) {
        super(delegate);
        if (alluxio.shaded.client.com.ressionLevel < 0 || alluxio.shaded.client.com.ressionLevel > 9) {
            throw new IllegalArgumentException("alluxio.shaded.client.com.ressionLevel: " + alluxio.shaded.client.com.ressionLevel + " (expected: 0-9)");
        }
        if (windowBits < 9 || windowBits > 15) {
            throw new IllegalArgumentException("windowBits: " + windowBits + " (expected: 9-15)");
        }
        if (memLevel < 1 || memLevel > 9) {
            throw new IllegalArgumentException("memLevel: " + memLevel + " (expected: 1-9)");
        }
        this.alluxio.shaded.client.com.ressionLevel = alluxio.shaded.client.com.ressionLevel;
        this.windowBits = windowBits;
        this.memLevel = memLevel;

        propertyKey = connection().newKey();
        connection().addListener(new Http2ConnectionAdapter() {
            @Override
            public void onStreamRemoved(Http2Stream stream) {
                final EmbeddedChannel alluxio.shaded.client.com.ressor = stream.getProperty(propertyKey);
                if (alluxio.shaded.client.com.ressor != null) {
                    cleanup(stream, alluxio.shaded.client.com.ressor);
                }
            }
        });
    }

    @Override
    public ChannelFuture writeData(final ChannelHandlerContext ctx, final int streamId, ByteBuf data, int padding,
            final boolean endOfStream, ChannelPromise promise) {
        final Http2Stream stream = connection().stream(streamId);
        final EmbeddedChannel channel = stream == null ? null : (EmbeddedChannel) stream.getProperty(propertyKey);
        if (channel == null) {
            // The alluxio.shaded.client.com.ressor may be null if no alluxio.shaded.client.com.atible encoding type was found in this stream's headers
            return super.writeData(ctx, streamId, data, padding, endOfStream, promise);
        }

        try {
            // The channel will release the buffer after being written
            channel.writeOutbound(data);
            ByteBuf buf = nextReadableBuf(channel);
            if (buf == null) {
                if (endOfStream) {
                    if (channel.finish()) {
                        buf = nextReadableBuf(channel);
                    }
                    return super.writeData(ctx, streamId, buf == null ? Unpooled.EMPTY_BUFFER : buf, padding,
                            true, promise);
                }
                // END_STREAM is not set and the assumption is data is still forthcoming.
                promise.setSuccess();
                return promise;
            }

            PromiseCombiner alluxio.shaded.client.com.iner = new PromiseCombiner();
            for (;;) {
                ByteBuf nextBuf = nextReadableBuf(channel);
                boolean alluxio.shaded.client.com.ressedEndOfStream = nextBuf == null && endOfStream;
                if (alluxio.shaded.client.com.ressedEndOfStream && channel.finish()) {
                    nextBuf = nextReadableBuf(channel);
                    alluxio.shaded.client.com.ressedEndOfStream = nextBuf == null;
                }

                ChannelPromise bufPromise = ctx.newPromise();
                alluxio.shaded.client.com.iner.add(bufPromise);
                super.writeData(ctx, streamId, buf, padding, alluxio.shaded.client.com.ressedEndOfStream, bufPromise);
                if (nextBuf == null) {
                    break;
                }

                padding = 0; // Padding is only alluxio.shaded.client.com.unicated once on the first iteration
                buf = nextBuf;
            }
            alluxio.shaded.client.com.iner.finish(promise);
        } catch (Throwable cause) {
            promise.tryFailure(cause);
        } finally {
            if (endOfStream) {
                cleanup(stream, channel);
            }
        }
        return promise;
    }

    @Override
    public ChannelFuture writeHeaders(ChannelHandlerContext ctx, int streamId, Http2Headers headers, int padding,
            boolean endStream, ChannelPromise promise) {
        try {
            // Determine if alluxio.shaded.client.com.ression is required and sanitize the headers.
            EmbeddedChannel alluxio.shaded.client.com.ressor = newCompressor(ctx, headers, endStream);

            // Write the headers and create the stream object.
            ChannelFuture future = super.writeHeaders(ctx, streamId, headers, padding, endStream, promise);

            // After the stream object has been created, then attach the alluxio.shaded.client.com.ressor as a property for data alluxio.shaded.client.com.ression.
            bindCompressorToStream(alluxio.shaded.client.com.ressor, streamId);

            return future;
        } catch (Throwable e) {
            promise.tryFailure(e);
        }
        return promise;
    }

    @Override
    public ChannelFuture writeHeaders(final ChannelHandlerContext ctx, final int streamId, final Http2Headers headers,
            final int streamDependency, final short weight, final boolean exclusive, final int padding,
            final boolean endOfStream, final ChannelPromise promise) {
        try {
            // Determine if alluxio.shaded.client.com.ression is required and sanitize the headers.
            EmbeddedChannel alluxio.shaded.client.com.ressor = newCompressor(ctx, headers, endOfStream);

            // Write the headers and create the stream object.
            ChannelFuture future = super.writeHeaders(ctx, streamId, headers, streamDependency, weight, exclusive,
                                                      padding, endOfStream, promise);

            // After the stream object has been created, then attach the alluxio.shaded.client.com.ressor as a property for data alluxio.shaded.client.com.ression.
            bindCompressorToStream(alluxio.shaded.client.com.ressor, streamId);

            return future;
        } catch (Throwable e) {
            promise.tryFailure(e);
        }
        return promise;
    }

    /**
     * Returns a new {@link EmbeddedChannel} that encodes the HTTP2 message content encoded in the specified
     * {@code contentEncoding}.
     *
     * @param ctx the context.
     * @param contentEncoding the value of the {@code content-encoding} header
     * @return a new {@link ByteToMessageDecoder} if the specified encoding is supported. {@code null} otherwise
     * (alternatively, you can throw a {@link Http2Exception} to block unknown encoding).
     * @throws Http2Exception If the specified encoding is not not supported and warrants an exception
     */
    protected EmbeddedChannel newContentCompressor(ChannelHandlerContext ctx, CharSequence contentEncoding)
            throws Http2Exception {
        if (GZIP.contentEqualsIgnoreCase(contentEncoding) || X_GZIP.contentEqualsIgnoreCase(contentEncoding)) {
            return newCompressionChannel(ctx, ZlibWrapper.GZIP);
        }
        if (DEFLATE.contentEqualsIgnoreCase(contentEncoding) || X_DEFLATE.contentEqualsIgnoreCase(contentEncoding)) {
            return newCompressionChannel(ctx, ZlibWrapper.ZLIB);
        }
        // 'identity' or unsupported
        return null;
    }

    /**
     * Returns the expected content encoding of the decoded content. Returning {@code contentEncoding} is the default
     * behavior, which is the case for most alluxio.shaded.client.com.ressors.
     *
     * @param contentEncoding the value of the {@code content-encoding} header
     * @return the expected content encoding of the new content.
     * @throws Http2Exception if the {@code contentEncoding} is not supported and warrants an exception
     */
    protected CharSequence getTargetContentEncoding(CharSequence contentEncoding) throws Http2Exception {
        return contentEncoding;
    }

    /**
     * Generate a new instance of an {@link EmbeddedChannel} capable of alluxio.shaded.client.com.ressing data
     * @param ctx the context.
     * @param wrapper Defines what type of encoder should be used
     */
    private EmbeddedChannel newCompressionChannel(final ChannelHandlerContext ctx, ZlibWrapper wrapper) {
        return new EmbeddedChannel(ctx.channel().id(), ctx.channel().metadata().hasDisconnect(),
                ctx.channel().config(), ZlibCodecFactory.newZlibEncoder(wrapper, alluxio.shaded.client.com.ressionLevel, windowBits,
                memLevel));
    }

    /**
     * Checks if a new alluxio.shaded.client.com.ressor object is needed for the stream identified by {@code streamId}. This method will
     * modify the {@code content-encoding} header contained in {@code headers}.
     *
     * @param ctx the context.
     * @param headers Object representing headers which are to be written
     * @param endOfStream Indicates if the stream has ended
     * @return The channel used to alluxio.shaded.client.com.ress data.
     * @throws Http2Exception if any problems occur during initialization.
     */
    private EmbeddedChannel newCompressor(ChannelHandlerContext ctx, Http2Headers headers, boolean endOfStream)
            throws Http2Exception {
        if (endOfStream) {
            return null;
        }

        CharSequence encoding = headers.get(CONTENT_ENCODING);
        if (encoding == null) {
            encoding = IDENTITY;
        }
        final EmbeddedChannel alluxio.shaded.client.com.ressor = newContentCompressor(ctx, encoding);
        if (alluxio.shaded.client.com.ressor != null) {
            CharSequence targetContentEncoding = getTargetContentEncoding(encoding);
            if (IDENTITY.contentEqualsIgnoreCase(targetContentEncoding)) {
                headers.remove(CONTENT_ENCODING);
            } else {
                headers.set(CONTENT_ENCODING, targetContentEncoding);
            }

            // The content length will be for the decompressed data. Since we will alluxio.shaded.client.com.ress the data
            // this content-length will not be correct. Instead of queuing messages or delaying sending
            // header frames...just remove the content-length header
            headers.remove(CONTENT_LENGTH);
        }

        return alluxio.shaded.client.com.ressor;
    }

    /**
     * Called after the super class has written the headers and created any associated stream objects.
     * @param alluxio.shaded.client.com.ressor The alluxio.shaded.client.com.ressor associated with the stream identified by {@code streamId}.
     * @param streamId The stream id for which the headers were written.
     */
    private void bindCompressorToStream(EmbeddedChannel alluxio.shaded.client.com.ressor, int streamId) {
        if (alluxio.shaded.client.com.ressor != null) {
            Http2Stream stream = connection().stream(streamId);
            if (stream != null) {
                stream.setProperty(propertyKey, alluxio.shaded.client.com.ressor);
            }
        }
    }

    /**
     * Release remaining content from {@link EmbeddedChannel} and remove the alluxio.shaded.client.com.ressor from the {@link Http2Stream}.
     *
     * @param stream The stream for which {@code alluxio.shaded.client.com.ressor} is the alluxio.shaded.client.com.ressor for
     * @param alluxio.shaded.client.com.ressor The alluxio.shaded.client.com.ressor for {@code stream}
     */
    void cleanup(Http2Stream stream, EmbeddedChannel alluxio.shaded.client.com.ressor) {
        if (alluxio.shaded.client.com.ressor.finish()) {
            for (;;) {
                final ByteBuf buf = alluxio.shaded.client.com.ressor.readOutbound();
                if (buf == null) {
                    break;
                }

                buf.release();
            }
        }
        stream.removeProperty(propertyKey);
    }

    /**
     * Read the next alluxio.shaded.client.com.ressed {@link ByteBuf} from the {@link EmbeddedChannel} or {@code null} if one does not exist.
     *
     * @param alluxio.shaded.client.com.ressor The channel to read from
     * @return The next decoded {@link ByteBuf} from the {@link EmbeddedChannel} or {@code null} if one does not exist
     */
    private static ByteBuf nextReadableBuf(EmbeddedChannel alluxio.shaded.client.com.ressor) {
        for (;;) {
            final ByteBuf buf = alluxio.shaded.client.com.ressor.readOutbound();
            if (buf == null) {
                return null;
            }
            if (!buf.isReadable()) {
                buf.release();
                continue;
            }
            return buf;
        }
    }
}
