/*
 * Decompiled with CFR 0.152.
 */
package alluxio.collections;

import alluxio.concurrent.LockMode;
import alluxio.resource.LockResource;
import alluxio.resource.RefCountLockResource;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.util.ThreadFactoryUtils;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LockPool<K> {
    private static final Logger LOG = LoggerFactory.getLogger(LockPool.class);
    private static final float DEFAULT_LOAD_FACTOR = 0.75f;
    private static final String EVICTOR_THREAD_NAME = "LockPool Evictor";
    private final Map<K, Resource> mPool;
    private final Function<? super K, ? extends ReentrantReadWriteLock> mDefaultLoader;
    private final int mLowWatermark;
    private final int mHighWatermark;
    private final Lock mEvictLock = new ReentrantLock();
    private final Condition mOverHighWatermark = this.mEvictLock.newCondition();
    private final ExecutorService mEvictor;

    public LockPool(Function<? super K, ? extends ReentrantReadWriteLock> defaultLoader, int initialSize, int lowWatermark, int highWatermark, int concurrencyLevel) {
        this.mDefaultLoader = defaultLoader;
        this.mLowWatermark = lowWatermark;
        this.mHighWatermark = highWatermark;
        this.mPool = new ConcurrentHashMap<K, Resource>(initialSize, 0.75f, concurrencyLevel);
        this.mEvictor = Executors.newSingleThreadExecutor(ThreadFactoryUtils.build(EVICTOR_THREAD_NAME, true));
        this.mEvictor.submit(new Evictor());
    }

    public LockResource get(K key, LockMode mode) {
        Resource resource = this.getResource(key);
        ReentrantReadWriteLock lock = resource.mLock;
        switch (mode) {
            case READ: {
                return new RefCountLockResource(lock.readLock(), true, resource.mRefCount);
            }
            case WRITE: {
                return new RefCountLockResource(lock.writeLock(), true, resource.mRefCount);
            }
        }
        throw new IllegalStateException("Unknown lock mode: " + (Object)((Object)mode));
    }

    public Optional<LockResource> tryGet(K key, LockMode mode) {
        Lock innerLock;
        Resource resource = this.getResource(key);
        ReentrantReadWriteLock lock = resource.mLock;
        switch (mode) {
            case READ: {
                innerLock = lock.readLock();
                break;
            }
            case WRITE: {
                innerLock = lock.writeLock();
                break;
            }
            default: {
                throw new IllegalStateException("Unknown lock mode: " + (Object)((Object)mode));
            }
        }
        if (!innerLock.tryLock()) {
            return Optional.empty();
        }
        return Optional.of(new RefCountLockResource(innerLock, false, resource.mRefCount));
    }

    @VisibleForTesting
    public ReentrantReadWriteLock getRawReadWriteLock(K key) {
        return this.mPool.getOrDefault(key, new Resource(new ReentrantReadWriteLock())).mLock;
    }

    private Resource getResource(K key) {
        Preconditions.checkNotNull(key, "key can not be null");
        Resource resource = this.mPool.compute(key, (k, v) -> {
            if (v != null && ((Resource)v).mRefCount.incrementAndGet() > 0) {
                ((Resource)v).mIsAccessed = true;
                return v;
            }
            return new Resource(this.mDefaultLoader.apply(k));
        });
        if (this.mPool.size() > this.mHighWatermark && this.mEvictLock.tryLock()) {
            try {
                this.mOverHighWatermark.signal();
            }
            finally {
                this.mEvictLock.unlock();
            }
        }
        return resource;
    }

    @VisibleForTesting
    public boolean containsKey(K key) {
        Preconditions.checkNotNull(key, "key can not be null");
        return this.mPool.containsKey(key);
    }

    public int size() {
        return this.mPool.size();
    }

    @VisibleForTesting
    public Map<K, ReentrantReadWriteLock> getEntryMap() {
        HashMap entries = new HashMap();
        this.mPool.entrySet().forEach(entry -> entries.put(entry.getKey(), ((Resource)entry.getValue()).mLock));
        return entries;
    }

    private static final class Resource {
        private final ReentrantReadWriteLock mLock;
        private volatile boolean mIsAccessed;
        private AtomicInteger mRefCount;

        private Resource(ReentrantReadWriteLock lock) {
            this.mLock = lock;
            this.mIsAccessed = false;
            this.mRefCount = new AtomicInteger(1);
        }
    }

    private final class Evictor
    implements Runnable {
        private static final long OVER_HIGH_WATERMARK_LOG_INTERVAL = 60000L;
        private static final int EVICTION_MAX_AWAIT_TIME = 30000;
        private long mLastSizeWarningTime = 0L;
        private Iterator<Map.Entry<K, Resource>> mIterator;

        public Evictor() {
            this.mIterator = LockPool.this.mPool.entrySet().iterator();
        }

        @Override
        public void run() {
            try {
                while (!Thread.interrupted()) {
                    this.awaitAndEvict();
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        private void awaitAndEvict() throws InterruptedException {
            try (LockResource l = new LockResource(LockPool.this.mEvictLock);){
                while (LockPool.this.mPool.size() <= LockPool.this.mHighWatermark) {
                    LockPool.this.mOverHighWatermark.await(30000L, TimeUnit.MILLISECONDS);
                }
                int numToEvict = LockPool.this.mPool.size() - LockPool.this.mLowWatermark;
                int roundToScan = 3;
                while (numToEvict > 0 && roundToScan > 0) {
                    Map.Entry candidateMapEntry;
                    Resource candidate;
                    if (!this.mIterator.hasNext()) {
                        this.mIterator = LockPool.this.mPool.entrySet().iterator();
                        --roundToScan;
                    }
                    if ((candidate = (candidateMapEntry = this.mIterator.next()).getValue()).mIsAccessed) {
                        candidate.mIsAccessed = false;
                        continue;
                    }
                    if (!candidate.mRefCount.compareAndSet(0, Integer.MIN_VALUE)) continue;
                    this.mIterator.remove();
                    --numToEvict;
                }
                if (LockPool.this.mPool.size() >= LockPool.this.mHighWatermark && System.currentTimeMillis() - this.mLastSizeWarningTime > 60000L) {
                    LOG.warn("LockPool size grows over high watermark: pool size = {}, low watermark = {}, high watermark = {}", new Object[]{LockPool.this.mPool.size(), LockPool.this.mLowWatermark, LockPool.this.mHighWatermark});
                    this.mLastSizeWarningTime = System.currentTimeMillis();
                }
            }
        }
    }
}

