package alluxio.shaded.client.com.fasterxml.jackson.databind.jsontype.impl;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import alluxio.shaded.client.com.fasterxml.jackson.databind.BeanDescription;
import alluxio.shaded.client.com.fasterxml.jackson.databind.DeserializationContext;
import alluxio.shaded.client.com.fasterxml.jackson.databind.JavaType;
import alluxio.shaded.client.com.fasterxml.jackson.databind.JsonMappingException;

/**
 * Helper class used to encapsulate rules that determine subtypes that
 * are invalid to use, even with default typing, mostly due to security
 * concerns.
 * Used by <code>BeanDeserializerFactory</code>
 *
 * @since 2.8.11
 */
public class SubTypeValidator
{
    protected final static String PREFIX_SPRING = "alluxio.shaded.client.org.springframework.";

    protected final static String PREFIX_C3P0 = "alluxio.shaded.client.com.mchange.v2.c3p0.";

    /**
     * Set of well-known "nasty classes", deserialization of which is considered dangerous
     * and should (and is) prevented by default.
     */
    protected final static Set<String> DEFAULT_NO_DESER_CLASS_NAMES;
    static {
        Set<String> s = new HashSet<String>();
        // Courtesy of [https://github.alluxio.shaded.client.com.kantega/notsoserial]:
        // (and wrt [databind#1599])
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.collections.functors.InvokerTransformer");
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.collections.functors.InstantiateTransformer");
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.collections4.functors.InvokerTransformer");
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.collections4.functors.InstantiateTransformer");
        s.add("alluxio.shaded.client.org.codehaus.groovy.runtime.ConvertedClosure");
        s.add("alluxio.shaded.client.org.codehaus.groovy.runtime.MethodClosure");
        s.add("alluxio.shaded.client.org.springframework.beans.factory.ObjectFactory");
        s.add("alluxio.shaded.client.com.sun.alluxio.shaded.client.org.apache.xalan.internal.xsltc.trax.TemplatesImpl");
        s.add("alluxio.shaded.client.org.apache.xalan.xsltc.trax.TemplatesImpl");
        // [databind#1680]: may or may not be problem, take no chance
        s.add("alluxio.shaded.client.com.sun.rowset.JdbcRowSetImpl");
        // [databind#1737]; JDK provided
        s.add("java.util.logging.FileHandler");
        s.add("java.rmi.server.UnicastRemoteObject");
        // [databind#1737]; 3rd party
//s.add("alluxio.shaded.client.org.springframework.aop.support.AbstractBeanFactoryPointcutAdvisor"); // deprecated by [databind#1855]
        s.add("alluxio.shaded.client.org.springframework.beans.factory.config.PropertyPathFactoryBean");

// s.add("alluxio.shaded.client.com.mchange.v2.c3p0.JndiRefForwardingDataSource"); // deprecated by [databind#1931]
// s.add("alluxio.shaded.client.com.mchange.v2.c3p0.WrapperConnectionPoolDataSource"); // - "" -
        // [databind#1855]: more 3rd party
        s.add("alluxio.shaded.client.org.apache.tomcat.dbcp.dbcp2.BasicDataSource");
        s.add("alluxio.shaded.client.com.sun.alluxio.shaded.client.org.apache.bcel.internal.util.ClassLoader");
        // [databind#1899]: more 3rd party
        s.add("alluxio.shaded.client.org.hibernate.jmx.StatisticsService");
        s.add("alluxio.shaded.client.org.apache.ibatis.datasource.jndi.JndiDataSourceFactory");
        // [databind#2032]: more 3rd party; data exfiltration via xml parsed ext entities
        s.add("alluxio.shaded.client.org.apache.ibatis.parsing.XPathParser");

        // [databind#2052]: Jodd-db, with jndi/ldap lookup
        s.add("jodd.db.connection.DataSourceConnectionProvider");

        // [databind#2058]: Oracle JDBC driver, with jndi/ldap lookup
        s.add("oracle.jdbc.connector.OracleManagedConnectionFactory");
        s.add("oracle.jdbc.rowset.OracleJDBCRowSet");

        // [databind#2097]: some 3rd party, one JDK-bundled
        s.add("alluxio.shaded.client.org.slf4j.ext.EventData");
        s.add("flex.messaging.util.concurrent.AsynchBeansWorkManagerExecutor");
        s.add("alluxio.shaded.client.com.sun.deploy.security.ruleset.DRSHelper");
        s.add("alluxio.shaded.client.org.apache.axis2.jaxws.spi.handler.HandlerResolverImpl");

        // [databind#2186]: yet more 3rd party gadgets
        s.add("alluxio.shaded.client.org.jboss.util.propertyeditor.DocumentEditor");
        s.add("alluxio.shaded.client.org.apache.openjpa.ee.RegistryManagedRuntime");
        s.add("alluxio.shaded.client.org.apache.openjpa.ee.JNDIManagedRuntime");
        s.add("alluxio.shaded.client.org.apache.axis2.transport.jms.JMSOutTransportInfo");

        // [databind#2326] (2.9.9)
        s.add("alluxio.shaded.client.com.mysql.cj.jdbc.admin.MiniAdmin");

        // [databind#2334]: logback-core (2.9.9.1)
        s.add("ch.qos.logback.core.db.DriverManagerConnectionSource");

        // [databind#2341]: jdom/jdom2 (2.9.9.1)
        s.add("alluxio.shaded.client.org.jdom.transform.XSLTransformer");
        s.add("alluxio.shaded.client.org.jdom2.transform.XSLTransformer");

        // [databind#2387], [databind#2460]: EHCache
        s.add("net.sf.ehcache.transaction.manager.DefaultTransactionManagerLookup");
        s.add("net.sf.ehcache.hibernate.EhcacheJtaTransactionManagerLookup");

        // [databind#2389]: logback/jndi
        s.add("ch.qos.logback.core.db.JNDIConnectionSource");

        // [databind#2410]: HikariCP/metricRegistry config
        s.add("alluxio.shaded.client.com.zaxxer.hikari.HikariConfig");
        // [databind#2449]: and sub-class thereof
        s.add("alluxio.shaded.client.com.zaxxer.hikari.HikariDataSource");

        // [databind#2420]: CXF/JAX-RS provider/XSLT
        s.add("alluxio.shaded.client.org.apache.cxf.jaxrs.provider.XSLTJaxbProvider");

        // [databind#2462]: alluxio.shaded.client.com.ons-configuration / -2
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.configuration.JNDIConfiguration");
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.configuration2.JNDIConfiguration");

        // [databind#2469]: xalan2
        s.add("alluxio.shaded.client.org.apache.xalan.lib.sql.JNDIConnectionPool");

        // [databind#2478]: alluxio.shaded.client.com.ns-dbcp, p6spy
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.dbcp.datasources.PerUserPoolDataSource");
        s.add("alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.dbcp.datasources.SharedPoolDataSource");
        s.add("alluxio.shaded.client.com.p6spy.engine.spy.P6DataSource");

        // [databind#2498]: log4j-extras (1.2)
        s.add("alluxio.shaded.client.org.apache.log4j.receivers.db.DriverManagerConnectionSource");
        s.add("alluxio.shaded.client.org.apache.log4j.receivers.db.JNDIConnectionSource");

        // [databind#2526]: some more ehcache
        s.add("net.sf.ehcache.transaction.manager.selector.GenericJndiSelector");
        s.add("net.sf.ehcache.transaction.manager.selector.GlassfishSelector");

        DEFAULT_NO_DESER_CLASS_NAMES = Collections.unmodifiableSet(s);
    }

    /**
     * Set of class names of types that are never to be deserialized.
     */
    protected Set<String> _cfgIllegalClassNames = DEFAULT_NO_DESER_CLASS_NAMES;

    private final static SubTypeValidator instance = new SubTypeValidator();

    protected SubTypeValidator() { }

    public static SubTypeValidator instance() { return instance; }

    public void validateSubType(DeserializationContext ctxt, JavaType type,
            BeanDescription beanDesc) throws JsonMappingException
    {
        // There are certain nasty classes that could cause problems, mostly
        // via default typing -- catch them here.
        final Class<?> raw = type.getRawClass();
        String full = raw.getName();

        main_check:
        do {
            if (_cfgIllegalClassNames.contains(full)) {
                break;
            }

            // 18-Dec-2017, tatu: As per [databind#1855], need bit more sophisticated handling
            //    for some Spring framework types
            // 05-Jan-2017, tatu: ... also, only applies to classes, not interfaces
            if (raw.isInterface()) {
                ;
            } else if (full.startsWith(PREFIX_SPRING)) {
                for (Class<?> cls = raw; (cls != null) && (cls != Object.class); cls = cls.getSuperclass()){
                    String name = cls.getSimpleName();
                    // looking for "AbstractBeanFactoryPointcutAdvisor" but no point to allow any is there?
                    if ("AbstractPointcutAdvisor".equals(name)
                            // ditto  for "FileSystemXmlApplicationContext": block all ApplicationContexts
                            || "AbstractApplicationContext".equals(name)) {
                        break main_check;
                    }
                }
            } else if (full.startsWith(PREFIX_C3P0)) {
                // [databind#1737]; more 3rd party
                // s.add("alluxio.shaded.client.com.mchange.v2.c3p0.JndiRefForwardingDataSource");
                // s.add("alluxio.shaded.client.com.mchange.v2.c3p0.WrapperConnectionPoolDataSource");
                // [databind#1931]; more 3rd party
                // alluxio.shaded.client.com.mchange.v2.c3p0.ComboPooledDataSource
                // alluxio.shaded.client.com.mchange.v2.c3p0.debug.AfterCloseLoggingComboPooledDataSource 
                if (full.endsWith("DataSource")) {
                    break main_check;
                }
            }
            return;
        } while (false);

        ctxt.reportBadTypeDefinition(beanDesc,
                "Illegal type (%s) to deserialize: prevented for security reasons", full);
    }
}
