/*
 * Copyright 2019 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.grpc.netty;

import alluxio.shaded.client.io.grpc.ChannelLogger;
import alluxio.shaded.client.io.grpc.netty.ProtocolNegotiators.ClientTlsHandler;
import alluxio.shaded.client.io.grpc.netty.ProtocolNegotiators.GrpcNegotiationHandler;
import alluxio.shaded.client.io.grpc.netty.ProtocolNegotiators.WaitUntilActiveHandler;
import alluxio.shaded.client.io.netty.channel.ChannelHandler;
import alluxio.shaded.client.io.netty.channel.ChannelHandlerContext;
import alluxio.shaded.client.io.netty.handler.ssl.SslContext;
import alluxio.shaded.client.io.netty.util.AsciiString;

/**
 * Internal accessor for {@link ProtocolNegotiators}.
 */
public final class InternalProtocolNegotiators {

  private InternalProtocolNegotiators() {}

  /**
   * Returns the channel logger for the given channel context, or a Noop Logger if absent.
   */
  public static ChannelLogger negotiationLogger(ChannelHandlerContext ctx) {
    return ProtocolNegotiators.negotiationLogger(ctx);
  }

  /**
   * Returns a {@link ProtocolNegotiator} that ensures the pipeline is set up so that TLS will
   * be negotiated, the {@code handler} is added and writes to the {@link alluxio.shaded.client.io.netty.channel.Channel}
   * may happen immediately, even before the TLS Handshake is alluxio.shaded.client.com.lete.
   */
  public static InternalProtocolNegotiator.ProtocolNegotiator tls(SslContext sslContext) {
    final alluxio.shaded.client.io.grpc.netty.ProtocolNegotiator negotiator = ProtocolNegotiators.tls(sslContext);
    final class TlsNegotiator implements InternalProtocolNegotiator.ProtocolNegotiator {

      @Override
      public AsciiString scheme() {
        return negotiator.scheme();
      }

      @Override
      public ChannelHandler newHandler(GrpcHttp2ConnectionHandler grpcHandler) {
        return negotiator.newHandler(grpcHandler);
      }

      @Override
      public void close() {
        negotiator.close();
      }
    }
    
    return new TlsNegotiator();
  }

  /**
   * Returns a {@link ProtocolNegotiator} that ensures the pipeline is set up so that TLS will be
   * negotiated, the server TLS {@code handler} is added and writes to the {@link
   * alluxio.shaded.client.io.netty.channel.Channel} may happen immediately, even before the TLS Handshake is alluxio.shaded.client.com.lete.
   */
  public static InternalProtocolNegotiator.ProtocolNegotiator serverTls(SslContext sslContext) {
    final alluxio.shaded.client.io.grpc.netty.ProtocolNegotiator negotiator = ProtocolNegotiators.serverTls(sslContext);
    final class ServerTlsNegotiator implements InternalProtocolNegotiator.ProtocolNegotiator {

      @Override
      public AsciiString scheme() {
        return negotiator.scheme();
      }

      @Override
      public ChannelHandler newHandler(GrpcHttp2ConnectionHandler grpcHandler) {
        return negotiator.newHandler(grpcHandler);
      }

      @Override
      public void close() {
        negotiator.close();
      }
    }

    return new ServerTlsNegotiator();
  }

  /** Returns a {@link ProtocolNegotiator} for plaintext client channel. */
  public static InternalProtocolNegotiator.ProtocolNegotiator plaintext() {
    final alluxio.shaded.client.io.grpc.netty.ProtocolNegotiator negotiator = ProtocolNegotiators.plaintext();
    final class PlaintextNegotiator implements InternalProtocolNegotiator.ProtocolNegotiator {

      @Override
      public AsciiString scheme() {
        return negotiator.scheme();
      }

      @Override
      public ChannelHandler newHandler(GrpcHttp2ConnectionHandler grpcHandler) {
        return negotiator.newHandler(grpcHandler);
      }

      @Override
      public void close() {
        negotiator.close();
      }
    }

    return new PlaintextNegotiator();
  }

  /** Returns a {@link ProtocolNegotiator} for plaintext server channel. */
  public static InternalProtocolNegotiator.ProtocolNegotiator serverPlaintext() {
    final alluxio.shaded.client.io.grpc.netty.ProtocolNegotiator negotiator = ProtocolNegotiators.serverPlaintext();
    final class ServerPlaintextNegotiator implements InternalProtocolNegotiator.ProtocolNegotiator {

      @Override
      public AsciiString scheme() {
        return negotiator.scheme();
      }

      @Override
      public ChannelHandler newHandler(GrpcHttp2ConnectionHandler grpcHandler) {
        return negotiator.newHandler(grpcHandler);
      }

      @Override
      public void close() {
        negotiator.close();
      }
    }

    return new ServerPlaintextNegotiator();
  }

  /**
   * Internal version of {@link WaitUntilActiveHandler}.
   */
  public static ChannelHandler waitUntilActiveHandler(ChannelHandler next) {
    return new WaitUntilActiveHandler(next);
  }

  /**
   * Internal version of {@link GrpcNegotiationHandler}.
   */
  public static ChannelHandler grpcNegotiationHandler(GrpcHttp2ConnectionHandler next) {
    return new GrpcNegotiationHandler(next);
  }

  public static ChannelHandler clientTlsHandler(
      ChannelHandler next, SslContext sslContext, String authority) {
    return new ClientTlsHandler(next, sslContext, authority, null);
  }

  public static class ProtocolNegotiationHandler
      extends ProtocolNegotiators.ProtocolNegotiationHandler {

    protected ProtocolNegotiationHandler(ChannelHandler next, String negotiatorName) {
      super(next, negotiatorName);
    }

    protected ProtocolNegotiationHandler(ChannelHandler next) {
      super(next);
    }
  }
}
