/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageInfo;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.RocksPageStoreOptions;
import alluxio.exception.PageNotFoundException;
import alluxio.proto.client.Cache;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.com.google.common.collect.Streams;
import alluxio.shaded.client.javax.annotation.Nullable;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import alluxio.shaded.client.org.rocksdb.Options;
import alluxio.shaded.client.org.rocksdb.RocksDB;
import alluxio.shaded.client.org.rocksdb.RocksDBException;
import alluxio.shaded.client.org.rocksdb.RocksIterator;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class RocksPageStore
implements PageStore {
    private static final Logger LOG = LoggerFactory.getLogger(RocksPageStore.class);
    private static final byte[] CONF_KEY = "CONF".getBytes();
    private static final double ROCKS_OVERHEAD_RATIO = 0.2;
    private final long mCapacity;
    private final RocksDB mDb;
    private final RocksPageStoreOptions mOptions;

    public static RocksPageStore create(RocksPageStoreOptions options) throws IOException {
        Preconditions.checkArgument(options.getMaxPageSize() > 0);
        RocksDB.loadLibrary();
        Options rocksOptions = new Options();
        rocksOptions.setCreateIfMissing(true);
        rocksOptions.setWriteBufferSize((long)options.getWriteBufferSize());
        rocksOptions.setCompressionType(options.getCompressionType());
        RocksDB db = null;
        try {
            Cache.PRocksPageStoreOptions persistedOptions;
            db = RocksDB.open((Options)rocksOptions, (String)options.getRootDir());
            byte[] confData = db.get(CONF_KEY);
            Cache.PRocksPageStoreOptions pOptions = options.toProto();
            if (confData != null && !(persistedOptions = Cache.PRocksPageStoreOptions.parseFrom(confData)).equals(pOptions)) {
                db.close();
                throw new IOException("Inconsistent configuration for RocksPageStore");
            }
            db.put(CONF_KEY, pOptions.toByteArray());
        }
        catch (RocksDBException e) {
            if (db != null) {
                db.close();
            }
            throw new IOException("Couldn't open rocksDB database", e);
        }
        return new RocksPageStore(options, db);
    }

    private RocksPageStore(RocksPageStoreOptions options, RocksDB rocksDB) {
        this.mOptions = options;
        this.mCapacity = (long)((double)options.getCacheSize() / 1.2);
        this.mDb = rocksDB;
    }

    @Override
    public void put(PageId pageId, byte[] page) throws IOException {
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId);
            this.mDb.put(key, page);
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to store page", e);
        }
    }

    @Override
    public ReadableByteChannel get(PageId pageId, int pageOffset) throws IOException, PageNotFoundException {
        Preconditions.checkArgument(pageOffset >= 0, "page offset should be non-negative");
        try {
            byte[] page = this.mDb.get(RocksPageStore.getKeyFromPageId(pageId));
            if (page == null) {
                throw new PageNotFoundException(new String(RocksPageStore.getKeyFromPageId(pageId)));
            }
            Preconditions.checkArgument(pageOffset <= page.length, "page offset %s exceeded page size %s", pageOffset, page.length);
            ByteArrayInputStream bais = new ByteArrayInputStream(page);
            bais.skip(pageOffset);
            return Channels.newChannel(bais);
        }
        catch (RocksDBException e) {
            throw new IOException("Failed to retrieve page", e);
        }
    }

    @Override
    public void delete(PageId pageId, long pageSize) throws PageNotFoundException {
        try {
            byte[] key = RocksPageStore.getKeyFromPageId(pageId);
            this.mDb.delete(key);
        }
        catch (RocksDBException e) {
            throw new PageNotFoundException("Failed to remove page", e);
        }
    }

    @Override
    public void close() {
        this.mDb.close();
    }

    private static byte[] getKeyFromPageId(PageId pageId) {
        byte[] fileId = pageId.getFileId().getBytes();
        ByteBuffer buf = ByteBuffer.allocate(8 + fileId.length);
        buf.putLong(pageId.getPageIndex());
        buf.put(fileId);
        return buf.array();
    }

    @Nullable
    private static PageId getPageIdFromKey(byte[] key) {
        if (key.length < 8) {
            return null;
        }
        ByteBuffer buf = ByteBuffer.wrap(key);
        long pageIndex = buf.getLong();
        String fileId = Charset.defaultCharset().decode(buf).toString();
        return new PageId(fileId, pageIndex);
    }

    @Override
    public Stream<PageInfo> getPages() {
        RocksIterator iter = this.mDb.newIterator();
        iter.seekToFirst();
        return (Stream)Streams.stream(new PageIterator(iter)).onClose(() -> ((RocksIterator)iter).close());
    }

    @Override
    public long getCacheSize() {
        return this.mCapacity;
    }

    private class PageIterator
    implements Iterator<PageInfo> {
        private final RocksIterator mIter;
        private PageInfo mValue;

        PageIterator(RocksIterator iter) {
            this.mIter = iter;
        }

        @Override
        public boolean hasNext() {
            return this.ensureValue() != null;
        }

        @Override
        public PageInfo next() {
            PageInfo value = this.ensureValue();
            if (value == null) {
                throw new NoSuchElementException();
            }
            this.mIter.next();
            this.mValue = null;
            return value;
        }

        @Nullable
        private PageInfo ensureValue() {
            if (this.mValue == null) {
                while (this.mIter.isValid()) {
                    PageId id = RocksPageStore.getPageIdFromKey(this.mIter.key());
                    long size = this.mIter.value().length;
                    if (id != null) {
                        this.mValue = new PageInfo(id, size);
                        break;
                    }
                    this.mIter.next();
                }
            }
            return this.mValue;
        }
    }
}

