/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.cli;

import alluxio.conf.InstancedConfiguration;
import alluxio.exception.status.InvalidArgumentException;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.alluxio.shaded.client.io.Closer;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.cli.CommandLine;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.lang.ArrayUtils;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.lang.StringUtils;
import alluxio.shaded.client.org.slf4j.Logger;
import alluxio.shaded.client.org.slf4j.LoggerFactory;

import java.alluxio.shaded.client.io.Closeable;
import java.alluxio.shaded.client.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import alluxio.shaded.client.javax.annotation.Nullable;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;

/**
 * Abstract class for handling alluxio.shaded.client.com.and line inputs.
 */
@NotThreadSafe
public abstract class AbstractShell implements Closeable {
  private static final Logger LOG = LoggerFactory.getLogger(AbstractShell.class);

  private Map<String, String[]> mCommandAlias;
  private Set<String> mUnstableAlias;
  private Map<String, Command> mCommands;
  protected InstancedConfiguration mConfiguration;
  protected Closer mCloser;

  /**
   * Creates a new instance of {@link AbstractShell}.
   *
   * @param alluxio.shaded.client.com.andAlias replacements for alluxio.shaded.client.com.ands
   * @param unstableAlias set of unstable aliases which may be removed in the future
   * @param conf Alluxio configuration
   */
  public AbstractShell(Map<String, String[]> alluxio.shaded.client.com.andAlias,
      Set<String> unstableAlias, InstancedConfiguration conf) {
    mCloser = Closer.create();
    mConfiguration = conf; // This needs to go first in case loadCommands() uses the reference to
    // the configuration
    mUnstableAlias = unstableAlias;
    mCommandAlias = alluxio.shaded.client.com.andAlias;
    mCommands = loadCommands();
    // Register all loaded alluxio.shaded.client.com.ands under closer.
    mCommands.values().stream().forEach((cmd) -> mCloser.register(cmd));
  }

  /**
   * Handles the specified shell alluxio.shaded.client.com.and request, displaying usage if the alluxio.shaded.client.com.and format is invalid.
   *
   * @param argv [] Array of arguments given by the user's input from the terminal
   * @return 0 if alluxio.shaded.client.com.and is successful, -1 if an error occurred
   */
  public int run(String... argv) {
    if (argv.length == 0) {
      printUsage();
      return -1;
    }

    // Sanity check on the number of arguments
    String cmd = argv[0];
    Command alluxio.shaded.client.com.and = mCommands.get(cmd);

    if (alluxio.shaded.client.com.and == null) {
      String[] replacementCmd = getReplacementCmd(cmd);
      if (replacementCmd == null) {
        // Unknown alluxio.shaded.client.com.and (we didn't find the cmd in our dict)
        System.err.println(String.format("%s is an unknown alluxio.shaded.client.com.and.", cmd));
        printUsage();
        return -1;
      } else {
        // Handle alluxio.shaded.client.com.and alias
        if (mUnstableAlias != null && mUnstableAlias.contains(cmd)) {
          String deprecatedMsg =
              String.format("WARNING: %s is not a stable CLI alluxio.shaded.client.com.and. It may be removed in the "
                      + "future. Use with caution in scripts. You may use '%s' instead.",
                  cmd, StringUtils.join(replacementCmd, " "));
          System.out.println(deprecatedMsg);
        }

        String[] replacementArgv =
            (String[]) ArrayUtils.addAll(replacementCmd, ArrayUtils.subarray(argv, 1, argv.length));
        return run(replacementArgv);
      }
    }

    // Find the inner-most alluxio.shaded.client.com.and and its argument line.
    CommandLine cmdline;
    try {
      String[] currArgs = Arrays.copyOf(argv, argv.length);
      while (alluxio.shaded.client.com.and.hasSubCommand()) {
        if (currArgs.length < 2) {
          throw new InvalidArgumentException("No sub-alluxio.shaded.client.com.and is specified");
        }
        if (!alluxio.shaded.client.com.and.getSubCommands().containsKey(currArgs[1])) {
          throw new InvalidArgumentException("Unknown sub-alluxio.shaded.client.com.and: " + currArgs[1]);
        }
        alluxio.shaded.client.com.and = alluxio.shaded.client.com.and.getSubCommands().get(currArgs[1]);
        if (currArgs.length >= 2) {
          currArgs = Arrays.copyOfRange(currArgs, 1, currArgs.length);
        }
      }
      currArgs = Arrays.copyOfRange(currArgs, 1, currArgs.length);

      cmdline = alluxio.shaded.client.com.and.parseAndValidateArgs(currArgs);
    } catch (InvalidArgumentException e) {
      // It outputs a prompt message when passing wrong args to CLI
      System.out.println(e.getMessage());
      System.out.println("Usage: " + alluxio.shaded.client.com.and.getUsage());
      System.out.println(alluxio.shaded.client.com.and.getDescription());
      LOG.error("Invalid arguments for alluxio.shaded.client.com.and {}:", alluxio.shaded.client.com.and.getCommandName(), e);
      return -1;
    }

    // Handle the alluxio.shaded.client.com.and
    try {
      return alluxio.shaded.client.com.and.run(cmdline);
    } catch (Exception e) {
      System.out.println(e.getMessage());
      LOG.error("Error running " + StringUtils.join(argv, " "), e);
      return -1;
    }
  }

  /**
   * @return all alluxio.shaded.client.com.ands provided by this shell
   */
  public Collection<Command> getCommands() {
    return mCommands.values();
  }

  @Override
  public void close() throws IOException {
    mCloser.close();
  }

  /**
   * Gets the replacement alluxio.shaded.client.com.and for alias.
   *
   * @param cmd the name of the alluxio.shaded.client.com.and
   * @return replacement alluxio.shaded.client.com.and if cmd is an alias
   */
  @Nullable
  private String[] getReplacementCmd(String cmd) {
    if (mCommandAlias == null || !mCommandAlias.containsKey(cmd)) {
      return null;
    }
    return mCommandAlias.get(cmd);
  }

  /**
   * @return name of the shell
   */
  protected abstract String getShellName();

  /**
   * Map structure: Command name => {@link Command} instance.
   *
   * @return a set of alluxio.shaded.client.com.ands which can be executed under this shell
   */
  protected abstract Map<String, Command> loadCommands();

  /**
   * Prints usage for all alluxio.shaded.client.com.ands.
   */
  protected void printUsage() {
    System.out.println("Usage: alluxio " + getShellName() + " [generic options]");
    SortedSet<String> sortedCmds = new TreeSet<>(mCommands.keySet());
    for (String cmd : sortedCmds) {
      System.out.format("%-60s%n", "\t [" + mCommands.get(cmd).getUsage() + "]");
    }
  }
}
