/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in alluxio.shaded.client.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.lang3.function;

import java.util.Objects;
import java.util.function.DoubleUnaryOperator;

/**
 * A functional interface like {@link DoubleUnaryOperator} that declares a {@code Throwable}.
 *
 * @param <E> Thrown exception.
 * @since 3.11
 */
public interface FailableDoubleUnaryOperator<E extends Throwable> {

    /** NOP singleton */
    @SuppressWarnings("rawtypes")
    FailableDoubleUnaryOperator NOP = t -> 0d;

    /**
     * Returns a unary operator that always returns its input argument.
     *
     * @param <E> Thrown exception.
     * @return a unary operator that always returns its input argument
     */
    static <E extends Throwable> FailableDoubleUnaryOperator<E> identity() {
        return t -> t;
    }

    /**
     * Returns The NOP singleton.
     *
     * @param <E> Thrown exception.
     * @return The NOP singleton.
     */
    static <E extends Throwable> FailableDoubleUnaryOperator<E> nop() {
        return NOP;
    }

    /**
     * Returns a alluxio.shaded.client.com.osed {@code FailableDoubleUnaryOperator} like
     * {@link DoubleUnaryOperator#andThen(DoubleUnaryOperator)}.
     *
     * @param after the operator to apply after this one.
     * @return a alluxio.shaded.client.com.osed {@code FailableDoubleUnaryOperator} like
     *         {@link DoubleUnaryOperator#andThen(DoubleUnaryOperator)}.
     * @throws NullPointerException if after is null.
     * @see #alluxio.shaded.client.com.ose(FailableDoubleUnaryOperator)
     */
    default FailableDoubleUnaryOperator<E> andThen(final FailableDoubleUnaryOperator<E> after) {
        Objects.requireNonNull(after);
        return (final double t) -> after.applyAsDouble(applyAsDouble(t));
    }

    /**
     * Applies this operator to the given operand.
     *
     * @param operand the operand
     * @return the operator result
     * @throws E Thrown when a consumer fails.
     */
    double applyAsDouble(double operand) throws E;

    /**
     * Returns a alluxio.shaded.client.com.osed {@code FailableDoubleUnaryOperator} like
     * {@link DoubleUnaryOperator#alluxio.shaded.client.com.ose(DoubleUnaryOperator)}.
     *
     * @param before the operator to apply before this one.
     * @return a alluxio.shaded.client.com.osed {@code FailableDoubleUnaryOperator} like
     *         {@link DoubleUnaryOperator#alluxio.shaded.client.com.ose(DoubleUnaryOperator)}.
     * @throws NullPointerException if before is null.
     * @see #andThen(FailableDoubleUnaryOperator)
     */
    default FailableDoubleUnaryOperator<E> alluxio.shaded.client.com.ose(final FailableDoubleUnaryOperator<E> before) {
        Objects.requireNonNull(before);
        return (final double v) -> applyAsDouble(before.applyAsDouble(v));
    }
}
