/*
 * Decompiled with CFR 0.152.
 */
package alluxio.collections;

import alluxio.concurrent.LockMode;
import alluxio.resource.LockResource;
import alluxio.resource.RWLockResource;
import alluxio.resource.RefCountLockResource;
import alluxio.resource.ResourcePool;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.MoreObjects;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.util.ThreadFactoryUtils;
import java.io.Closeable;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LockPool<K>
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(LockPool.class);
    private static final float DEFAULT_LOAD_FACTOR = 0.75f;
    private static final String EVICTOR_THREAD_NAME = "LockPool Evictor";
    private final Map<K, Resource> mPool;
    private final ResourcePool<Resource> mResueLockPool = new ResourcePool(1000000){

        @Override
        public void close() {
        }

        protected Resource createNewResource() {
            return new Resource((ReentrantReadWriteLock)LockPool.this.mDefaultLoader.get());
        }
    };
    private final Supplier<? extends ReentrantReadWriteLock> mDefaultLoader;
    private final int mLowWatermark;
    private final int mHighWatermark;
    private final Lock mEvictLock = new ReentrantLock();
    private final Condition mOverHighWatermark = this.mEvictLock.newCondition();
    private final ExecutorService mEvictor;
    private final Future<?> mEvictorTask;

    public LockPool(Supplier<? extends ReentrantReadWriteLock> defaultLoader, int initialSize, int lowWatermark, int highWatermark, int concurrencyLevel) {
        this.mDefaultLoader = defaultLoader;
        this.mLowWatermark = lowWatermark;
        this.mHighWatermark = highWatermark;
        this.mPool = new ConcurrentHashMap<K, Resource>(initialSize, 0.75f, concurrencyLevel);
        this.mEvictor = Executors.newSingleThreadExecutor(ThreadFactoryUtils.build(String.format("%s-%s", EVICTOR_THREAD_NAME, this.toString()), true));
        this.mEvictorTask = this.mEvictor.submit(new Evictor());
    }

    @Override
    public void close() throws IOException {
        this.mEvictorTask.cancel(true);
        this.mEvictor.shutdownNow();
        try {
            this.mEvictor.awaitTermination(2L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Failed to await LockPool evictor termination", e);
        }
    }

    public LockResource get(K key, LockMode mode) {
        return this.get(key, mode, false);
    }

    public RWLockResource get(K key, LockMode mode, boolean useTryLock) {
        Resource resource = this.getResource(key);
        return new RefCountLockResource(resource.mLock, mode, true, resource.mRefCount, useTryLock);
    }

    public Optional<RWLockResource> tryGet(K key, LockMode mode) {
        Lock innerLock;
        Resource resource = this.getResource(key);
        ReentrantReadWriteLock lock = resource.mLock;
        switch (mode) {
            case READ: {
                innerLock = lock.readLock();
                break;
            }
            case WRITE: {
                innerLock = lock.writeLock();
                break;
            }
            default: {
                throw new IllegalStateException("Unknown lock mode: " + (Object)((Object)mode));
            }
        }
        if (!innerLock.tryLock()) {
            return Optional.empty();
        }
        return Optional.of(new RefCountLockResource(lock, mode, false, resource.mRefCount, false));
    }

    @VisibleForTesting
    public ReentrantReadWriteLock getRawReadWriteLock(K key) {
        return this.mPool.getOrDefault(key, new Resource(new ReentrantReadWriteLock())).mLock;
    }

    private Resource getResource(K key) {
        Preconditions.checkNotNull(key, "key can not be null");
        Resource resource = this.mPool.compute(key, (k, v) -> {
            if (v != null && ((Resource)v).mRefCount.incrementAndGet() > 0) {
                ((Resource)v).mIsAccessed = true;
                return v;
            }
            Resource res = this.mResueLockPool.acquireWithoutBlocking();
            if (res == null) {
                res = new Resource(this.mDefaultLoader.get());
            }
            res.reset();
            return res;
        });
        if (this.mPool.size() > this.mHighWatermark && this.mEvictLock.tryLock()) {
            try {
                this.mOverHighWatermark.signal();
            }
            finally {
                this.mEvictLock.unlock();
            }
        }
        return resource;
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("lowWatermark", this.mLowWatermark).add("highWatermark", this.mHighWatermark).add("size", this.mPool.size()).toString();
    }

    @VisibleForTesting
    public boolean containsKey(K key) {
        Preconditions.checkNotNull(key, "key can not be null");
        return this.mPool.containsKey(key);
    }

    public int size() {
        return this.mPool.size();
    }

    @VisibleForTesting
    public Map<K, ReentrantReadWriteLock> getEntryMap() {
        HashMap entries = new HashMap();
        this.mPool.forEach((key, value) -> entries.put(key, ((Resource)value).mLock));
        return entries;
    }

    private static final class Resource {
        private final ReentrantReadWriteLock mLock;
        private volatile boolean mIsAccessed;
        private AtomicInteger mRefCount;

        private Resource(ReentrantReadWriteLock lock) {
            this.mLock = lock;
            this.mIsAccessed = false;
            this.mRefCount = new AtomicInteger(1);
        }

        public void reset() {
            this.mRefCount.set(1);
        }
    }

    private final class Evictor
    implements Runnable {
        private static final long OVER_HIGH_WATERMARK_LOG_INTERVAL = 60000L;
        private static final int EVICTION_MAX_AWAIT_TIME = 30000;
        private long mLastSizeWarningTime = 0L;
        private Iterator<Map.Entry<K, Resource>> mIterator;

        public Evictor() {
            this.mIterator = LockPool.this.mPool.entrySet().iterator();
        }

        @Override
        public void run() {
            try {
                while (!Thread.interrupted()) {
                    this.awaitAndEvict();
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        private void awaitAndEvict() throws InterruptedException {
            try (LockResource l = new LockResource(LockPool.this.mEvictLock);){
                while (LockPool.this.mPool.size() <= LockPool.this.mHighWatermark) {
                    LockPool.this.mOverHighWatermark.await(30000L, TimeUnit.MILLISECONDS);
                }
                int numToEvict = LockPool.this.mPool.size() - LockPool.this.mLowWatermark;
                int roundToScan = 3;
                while (numToEvict > 0 && roundToScan > 0) {
                    Map.Entry candidateMapEntry;
                    Resource candidate;
                    if (!this.mIterator.hasNext()) {
                        this.mIterator = LockPool.this.mPool.entrySet().iterator();
                        --roundToScan;
                    }
                    if ((candidate = (candidateMapEntry = this.mIterator.next()).getValue()).mIsAccessed) {
                        candidate.mIsAccessed = false;
                        continue;
                    }
                    if (!candidate.mRefCount.compareAndSet(0, Integer.MIN_VALUE)) continue;
                    this.mIterator.remove();
                    --numToEvict;
                    LockPool.this.mResueLockPool.release(candidate);
                }
                if (LockPool.this.mPool.size() >= LockPool.this.mHighWatermark && System.currentTimeMillis() - this.mLastSizeWarningTime > 60000L) {
                    LOG.warn("LockPool size grows over high watermark: pool size = {}, low watermark = {}, high watermark = {}", new Object[]{LockPool.this.mPool.size(), LockPool.this.mLowWatermark, LockPool.this.mHighWatermark});
                    this.mLastSizeWarningTime = System.currentTimeMillis();
                }
            }
        }
    }
}

