package alluxio.shaded.client.org.reflections.serializers;

import alluxio.shaded.client.com.google.gson.Gson;
import alluxio.shaded.client.com.google.gson.GsonBuilder;
import alluxio.shaded.client.org.reflections.Reflections;
import alluxio.shaded.client.org.reflections.util.Utils;

import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.file.Files;

/** serialization of Reflections to json
 *
 * <p>an example of produced json:
 * <pre>
 * {"store":{"storeMap":
 *    {"alluxio.shaded.client.org.reflections.scanners.TypeAnnotationsScanner":{
 *       "alluxio.shaded.client.org.reflections.TestModel$AC1":["alluxio.shaded.client.org.reflections.TestModel$C1"],
 *       "alluxio.shaded.client.org.reflections.TestModel$AC2":["alluxio.shaded.client.org.reflections.TestModel$I3",
 * ...
 * </pre>
 * */
public class JsonSerializer implements Serializer {
    private Gson gson;

    public Reflections read(InputStream inputStream) {
        return getGson().fromJson(new InputStreamReader(inputStream), Reflections.class);
    }

    public File save(Reflections reflections, String filename) {
        try {
            File file = Utils.prepareFile(filename);
            Files.write(file.toPath(), toString(reflections).getBytes(Charset.defaultCharset()));
            return file;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String toString(Reflections reflections) {
        return getGson().toJson(reflections);
    }

    private Gson getGson() {
        if (gson == null) {
            gson = new GsonBuilder().setPrettyPrinting().create();
        }
        return gson;
    }
}
