/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util.io;

import alluxio.AlluxioURI;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.InvalidPathException;
import alluxio.shaded.client.com.google.common.base.CharMatcher;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.javax.annotation.concurrent.ThreadSafe;
import alluxio.shaded.client.org.apache.commons.io.FilenameUtils;
import alluxio.util.OSUtils;
import java.util.Arrays;
import java.util.Collection;
import java.util.UUID;
import java.util.regex.Pattern;

@ThreadSafe
public final class PathUtils {
    private static final String TEMPORARY_SUFFIX_FORMAT = ".alluxio.0x%016X.tmp";
    private static final int TEMPORARY_SUFFIX_LENGTH = String.format(".alluxio.0x%016X.tmp", 0).length();
    private static final CharMatcher SEPARATOR_MATCHER = CharMatcher.is("/".charAt(0));
    private static final Pattern TEMPORARY_FILE_NAME = Pattern.compile("^.*\\.alluxio\\.0x[0-9A-F]{16}\\.tmp$");

    public static String cleanPath(String path) throws InvalidPathException {
        PathUtils.validatePath(path);
        return FilenameUtils.separatorsToUnix(FilenameUtils.normalizeNoEndSeparator(path));
    }

    public static String concatPath(Object base, Object path) {
        Preconditions.checkNotNull(base, "base");
        Preconditions.checkNotNull(path, "path");
        String trimmedBase = SEPARATOR_MATCHER.trimTrailingFrom(base.toString());
        String trimmedPath = SEPARATOR_MATCHER.trimFrom(path.toString());
        StringBuilder output = new StringBuilder(trimmedBase.length() + trimmedPath.length() + 1);
        output.append(trimmedBase);
        if (!trimmedPath.isEmpty()) {
            output.append("/");
            output.append(trimmedPath);
        }
        if (output.length() == 0) {
            return "/";
        }
        return output.toString();
    }

    public static String concatPath(Object base, Object ... paths) throws IllegalArgumentException {
        Preconditions.checkArgument(base != null, "Failed to concatPath: base is null");
        Preconditions.checkArgument(paths != null, "Failed to concatPath: a null set of paths");
        String trimmedBase = SEPARATOR_MATCHER.trimTrailingFrom(base.toString());
        StringBuilder output = new StringBuilder();
        output.append(trimmedBase);
        for (Object path : paths) {
            String trimmedPath;
            if (path == null || (trimmedPath = SEPARATOR_MATCHER.trimFrom(path.toString())).isEmpty()) continue;
            output.append("/");
            output.append(trimmedPath);
        }
        if (output.length() == 0) {
            return "/";
        }
        return output.toString();
    }

    public static AlluxioURI findLowestCommonAncestor(Collection<AlluxioURI> paths) {
        if (paths == null || paths.isEmpty()) {
            return null;
        }
        String[] matchedComponents = null;
        int matchedLen = 0;
        for (AlluxioURI path : paths) {
            String[] pathComp = path.getPath().split("/");
            if (matchedComponents == null) {
                matchedComponents = pathComp;
                matchedLen = pathComp.length;
                continue;
            }
            for (int i = 0; i < pathComp.length && i < matchedLen; ++i) {
                if (matchedComponents[i].equals(pathComp[i])) continue;
                matchedLen = i;
                break;
            }
            if (matchedLen <= pathComp.length) continue;
            matchedLen = pathComp.length;
        }
        return new AlluxioURI(PathUtils.concatPath((Object)"/", Arrays.copyOf(matchedComponents, matchedLen)));
    }

    public static String getParent(String path) throws InvalidPathException {
        return PathUtils.getParentCleaned(PathUtils.cleanPath(path));
    }

    public static String getParentCleaned(String cleanedPath) throws InvalidPathException {
        String name = FilenameUtils.getName(cleanedPath);
        String parent = cleanedPath.substring(0, cleanedPath.length() - name.length() - 1);
        if (parent.isEmpty()) {
            return "/";
        }
        return parent;
    }

    public static String concatUfsPath(String base, String path) {
        Pattern basePattern = Pattern.compile("(...)\\/\\/");
        if (!basePattern.matcher(base).matches()) {
            return PathUtils.concatPath((Object)base, (Object)path);
        }
        Preconditions.checkArgument(base != null, "Failed to concatPath: base is null");
        Preconditions.checkArgument(path != null, "Failed to concatPath: a null set of paths");
        String trimmedPath = SEPARATOR_MATCHER.trimFrom(path);
        StringBuilder output = new StringBuilder(base.length() + trimmedPath.length());
        output.append(base);
        output.append(trimmedPath);
        return output.toString();
    }

    public static String getPersistentTmpPath(AlluxioConfiguration ufsConfiguration, String path) {
        StringBuilder tempFilePath = new StringBuilder();
        StringBuilder tempFileName = new StringBuilder();
        String fileName = FilenameUtils.getName(path);
        String timeStamp = String.valueOf(System.currentTimeMillis());
        String uuid = UUID.randomUUID().toString();
        String tempDir = ufsConfiguration.getString(PropertyKey.UNDERFS_PERSISTENCE_ASYNC_TEMP_DIR);
        tempFilePath.append(tempDir);
        tempFilePath.append("/");
        tempFileName.append(fileName);
        tempFileName.append(".alluxio.");
        tempFileName.append(timeStamp);
        tempFileName.append(".");
        tempFileName.append(uuid);
        tempFileName.append(".tmp");
        tempFilePath.append((CharSequence)tempFileName);
        return tempFilePath.toString();
    }

    public static String[] getPathComponents(String path) throws InvalidPathException {
        if (PathUtils.isRoot(path = PathUtils.cleanPath(path))) {
            return new String[]{""};
        }
        return path.split("/");
    }

    public static String[] getCleanedPathComponents(String path) throws InvalidPathException {
        if (path == null || path.isEmpty()) {
            throw new InvalidPathException(ExceptionMessage.PATH_INVALID.getMessage(path));
        }
        if ("/".equals(path)) {
            return new String[]{""};
        }
        return path.split("/");
    }

    public static String subtractPaths(String path, String prefix) throws InvalidPathException {
        String cleanedPrefix;
        String cleanedPath = PathUtils.cleanPath(path);
        if (cleanedPath.equals(cleanedPrefix = PathUtils.cleanPath(prefix))) {
            return "";
        }
        if (!PathUtils.hasPrefix(cleanedPath, cleanedPrefix)) {
            throw new RuntimeException(String.format("Cannot subtract %s from %s because it is not a prefix", prefix, path));
        }
        int prefixLen = cleanedPrefix.length();
        int charsToDrop = PathUtils.isRoot(cleanedPrefix) ? prefixLen : prefixLen + 1;
        return cleanedPath.substring(charsToDrop, cleanedPath.length());
    }

    public static boolean hasPrefix(String path, String prefix) throws InvalidPathException {
        path = PathUtils.cleanPath(path);
        if ((prefix = PathUtils.cleanPath(prefix)).equals("/")) {
            return true;
        }
        if (!path.startsWith(prefix)) {
            return false;
        }
        return path.length() == prefix.length() || prefix.endsWith("/") || path.charAt(prefix.length()) == '/';
    }

    public static boolean isRoot(String path) throws InvalidPathException {
        return "/".equals(PathUtils.cleanPath(path));
    }

    public static void validatePath(String path) throws InvalidPathException {
        boolean invalid;
        boolean bl = invalid = path == null || path.isEmpty();
        if (!OSUtils.isWindows()) {
            boolean bl2 = invalid = invalid || !path.startsWith("/");
        }
        if (invalid) {
            throw new InvalidPathException(ExceptionMessage.PATH_INVALID.getMessage(path));
        }
    }

    public static String temporaryFileName(long nonce, String path) {
        return path + String.format(TEMPORARY_SUFFIX_FORMAT, nonce);
    }

    public static String getPermanentFileName(String path) {
        if (PathUtils.isTemporaryFileName(path)) {
            return path.substring(0, path.length() - TEMPORARY_SUFFIX_LENGTH);
        }
        return path;
    }

    public static boolean isTemporaryFileName(String path) {
        return TEMPORARY_FILE_NAME.matcher(path).matches();
    }

    public static String uniqPath() {
        StackTraceElement caller = new Throwable().getStackTrace()[1];
        long time = System.nanoTime();
        return "/" + caller.getClassName() + "/" + caller.getMethodName() + "/" + time;
    }

    public static String normalizePath(String path, String separator) {
        return path.endsWith(separator) ? path : path + separator;
    }

    private PathUtils() {
    }
}

