/*
 * Decompiled with CFR 0.152.
 */
package alluxio.extensions;

import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.extensions.ExtensionFactory;
import alluxio.extensions.ExtensionsClassLoader;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import alluxio.util.ExtensionUtils;
import alluxio.util.io.PathUtils;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.ServiceLoader;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class ExtensionFactoryRegistry<T extends ExtensionFactory<?, S>, S extends AlluxioConfiguration> {
    private static final Logger LOG = LoggerFactory.getLogger(ExtensionFactoryRegistry.class);
    private final List<T> mFactories = new CopyOnWriteArrayList<T>();
    private final String mExtensionPattern;
    private final Class<T> mFactoryClass;
    private boolean mInit = false;

    public ExtensionFactoryRegistry(Class<T> factoryClass, String extensionPattern) {
        this.mFactoryClass = Preconditions.checkNotNull(factoryClass, "factoryClass");
        this.mExtensionPattern = extensionPattern;
        this.init();
    }

    private synchronized void init() {
        ServiceLoader<ExtensionFactory> discoveredFactories = ServiceLoader.load(this.mFactoryClass, this.mFactoryClass.getClassLoader());
        for (ExtensionFactory factory : discoveredFactories) {
            LOG.debug("Discovered base extension factory implementation {} - {}", factory.getClass(), (Object)factory);
            this.register(factory);
        }
    }

    public List<T> getAvailable() {
        return Collections.unmodifiableList(this.mFactories);
    }

    public List<T> findAll(String path, S conf) {
        Preconditions.checkArgument(path != null, "path may not be null");
        List<T> eligibleFactories = this.scanRegistered(path, conf);
        if (!eligibleFactories.isEmpty()) {
            LOG.debug("Find {} eligible items from registered factories for path {}", (Object)eligibleFactories.size(), (Object)path);
            return eligibleFactories;
        }
        ArrayList<T> factories = new ArrayList<T>(this.mFactories);
        String libDir = PathUtils.concatPath((Object)conf.getString(PropertyKey.HOME), (Object)"lib");
        String extensionDir = conf.getString(PropertyKey.EXTENSIONS_DIR);
        this.scanLibs(factories, libDir);
        this.scanExtensions(factories, extensionDir);
        for (ExtensionFactory factory : factories) {
            if (!factory.supportsPath(path, conf)) continue;
            LOG.debug("Factory implementation {} is eligible for path {}", (Object)factory, (Object)path);
            eligibleFactories.add(factory);
        }
        if (eligibleFactories.isEmpty()) {
            LOG.warn("No factory implementation supports the path {}", (Object)path);
        }
        return eligibleFactories;
    }

    public List<T> scanRegistered(String path, S conf) {
        return this.mFactories.stream().filter(factory -> factory.supportsPath(path, conf)).collect(Collectors.toList());
    }

    private void scanExtensions(List<T> factories, String extensionsDir) {
        LOG.info("Loading extension jars from {}", (Object)extensionsDir);
        this.scan(Arrays.asList(ExtensionUtils.listExtensions(extensionsDir)), factories);
    }

    private void scanLibs(List<T> factories, String libDir) {
        LOG.info("Loading core jars from {}", (Object)libDir);
        ArrayList<File> files = new ArrayList<File>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(Paths.get(libDir, new String[0]), this.mExtensionPattern);){
            for (Path entry : stream) {
                if (!entry.toFile().isFile()) continue;
                files.add(entry.toFile());
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to load libs: {}", (Object)e.toString());
        }
        this.scan(files, factories);
    }

    private void scan(List<File> files, List<T> factories) {
        for (File jar : files) {
            try {
                URL extensionURL = jar.toURI().toURL();
                String jarPath = extensionURL.toString();
                ExtensionsClassLoader extensionsClassLoader = new ExtensionsClassLoader(new URL[]{extensionURL}, ClassLoader.getSystemClassLoader());
                ServiceLoader<ExtensionFactory> extensionServiceLoader = ServiceLoader.load(this.mFactoryClass, extensionsClassLoader);
                for (ExtensionFactory factory : extensionServiceLoader) {
                    LOG.debug("Discovered a factory implementation {} - {} in jar {}", new Object[]{factory.getClass(), factory, jarPath});
                    this.register(factory, factories);
                    this.register(factory);
                }
            }
            catch (Throwable t) {
                LOG.warn("Failed to load jar {}: {}", (Object)jar, (Object)t.toString());
            }
        }
    }

    public void register(T factory) {
        this.register(factory, this.mFactories);
    }

    private void register(T factory, List<T> factories) {
        if (factory == null) {
            return;
        }
        LOG.debug("Registered factory implementation {} - {}", factory.getClass(), factory);
        factories.add(0, factory);
    }

    public synchronized void reset() {
        if (this.mInit) {
            this.mInit = false;
            this.mFactories.clear();
        }
        this.init();
    }

    public void unregister(T factory) {
        this.unregister(factory, this.mFactories);
    }

    private void unregister(T factory, List<T> factories) {
        if (factory == null) {
            return;
        }
        LOG.debug("Unregistered factory implementation {} - {}", factory.getClass(), factory);
        factories.remove(factory);
    }
}

