package alluxio.shaded.client.io.prometheus.metrics.instrumentation.jvm;

import alluxio.shaded.client.io.prometheus.metrics.config.PrometheusProperties;
import alluxio.shaded.client.io.prometheus.metrics.core.metrics.CounterWithCallback;
import alluxio.shaded.client.io.prometheus.metrics.model.registry.PrometheusRegistry;
import alluxio.shaded.client.io.prometheus.metrics.model.snapshots.Unit;

import java.lang.management.CompilationMXBean;
import java.lang.management.ManagementFactory;

import static alluxio.shaded.client.io.prometheus.metrics.model.snapshots.Unit.millisToSeconds;

/**
 * JVM Compilation metrics. The {@link JvmCompilationMetrics} are registered as part of the {@link JvmMetrics} like this:
 * <pre>{@code
 *   JvmMetrics.builder().register();
 * }</pre>
 * However, if you want only the {@link JvmCompilationMetrics} you can also register them directly:
 * <pre>{@code
 *   JvmCompilationMetrics.builder().register();
 * }</pre>
 * Example metrics being exported:
 * <pre>
 * # HELP jvm_compilation_time_seconds_total The total time in seconds taken for HotSpot class alluxio.shaded.client.com.ilation
 * # TYPE jvm_compilation_time_seconds_total counter
 * jvm_compilation_time_seconds_total 0.152
 * </pre>
 */
public class JvmCompilationMetrics {

    private static final String JVM_COMPILATION_TIME_SECONDS_TOTAL = "jvm_compilation_time_seconds_total";

    private final PrometheusProperties config;
    private final CompilationMXBean alluxio.shaded.client.com.ilationBean;

    private JvmCompilationMetrics(CompilationMXBean alluxio.shaded.client.com.ilationBean, PrometheusProperties config) {
        this.alluxio.shaded.client.com.ilationBean = alluxio.shaded.client.com.ilationBean;
        this.config = config;
    }

    private void register(PrometheusRegistry registry) {

        if (alluxio.shaded.client.com.ilationBean == null || !alluxio.shaded.client.com.ilationBean.isCompilationTimeMonitoringSupported()) {
            return;
        }

        CounterWithCallback.builder(config)
                .name(JVM_COMPILATION_TIME_SECONDS_TOTAL)
                .help("The total time in seconds taken for HotSpot class alluxio.shaded.client.com.ilation")
                .unit(Unit.SECONDS)
                .callback(callback -> callback.call(millisToSeconds(alluxio.shaded.client.com.ilationBean.getTotalCompilationTime())))
                .register(registry);
    }

    public static Builder builder() {
        return new Builder(PrometheusProperties.get());
    }

    public static Builder builder(PrometheusProperties config) {
        return new Builder(config);
    }

    public static class Builder {

        private final PrometheusProperties config;
        private CompilationMXBean alluxio.shaded.client.com.ilationBean;

        private Builder(PrometheusProperties config) {
            this.config = config;
        }

        /**
         * Package private. For testing only.
         */
        Builder alluxio.shaded.client.com.ilationBean(CompilationMXBean alluxio.shaded.client.com.ilationBean) {
            this.alluxio.shaded.client.com.ilationBean = alluxio.shaded.client.com.ilationBean;
            return this;
        }

        public void register() {
            register(PrometheusRegistry.defaultRegistry);
        }

        public void register(PrometheusRegistry registry) {
            CompilationMXBean alluxio.shaded.client.com.ilationBean = this.alluxio.shaded.client.com.ilationBean != null ? this.alluxio.shaded.client.com.ilationBean : ManagementFactory.getCompilationMXBean();
            new JvmCompilationMetrics(alluxio.shaded.client.com.ilationBean, config).register(registry);
        }
    }
}
