/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.PageReadTargetBuffer;
import alluxio.client.file.cache.store.PageStoreDir;
import alluxio.client.file.cache.store.PageStoreOptions;
import alluxio.exception.PageNotFoundException;
import alluxio.exception.status.ResourceExhaustedException;
import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import alluxio.shaded.client.org.apache.commons.io.FileUtils;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;

@NotThreadSafe
public class LocalPageStore
implements PageStore {
    private static final String ERROR_NO_SPACE_LEFT = "No space left on device";
    public static final String TEMP_DIR = "TEMP";
    private final Path mRoot;
    private final long mPageSize;
    private final long mCapacity;
    private final int mFileBuckets;

    public LocalPageStore(PageStoreOptions options) {
        this.mRoot = options.getRootDir();
        this.mPageSize = options.getPageSize();
        this.mCapacity = (long)((double)options.getCacheSize() / (1.0 + options.getOverheadRatio()));
        this.mFileBuckets = options.getFileBuckets();
    }

    @Override
    public void put(PageId pageId, ByteBuffer page, boolean isTemporary) throws ResourceExhaustedException, IOException {
        Path pagePath = this.getPagePath(pageId, isTemporary);
        try {
            if (!Files.exists(pagePath, new LinkOption[0])) {
                Path parent = Preconditions.checkNotNull(pagePath.getParent(), "parent of cache file should not be null");
                Files.createDirectories(parent, new FileAttribute[0]);
                Files.createFile(pagePath, new FileAttribute[0]);
            }
            try (FileOutputStream fos = new FileOutputStream(pagePath.toFile(), false);){
                fos.getChannel().write(page);
            }
        }
        catch (Exception e) {
            Files.deleteIfExists(pagePath);
            if (e.getMessage().contains(ERROR_NO_SPACE_LEFT)) {
                throw new ResourceExhaustedException(String.format("%s is full, configured with %d bytes", this.mRoot, this.mCapacity), e);
            }
            throw new IOException("Failed to write file " + pagePath + " for page " + pageId);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public int get(PageId pageId, int pageOffset, int bytesToRead, PageReadTargetBuffer target, boolean isTemporary) throws IOException, PageNotFoundException {
        Preconditions.checkArgument(pageOffset >= 0, "page offset should be non-negative");
        Path pagePath = this.getPagePath(pageId, isTemporary);
        try (RandomAccessFile localFile = new RandomAccessFile(pagePath.toString(), "r");){
            int bytes;
            int bytesSkipped = localFile.skipBytes(pageOffset);
            if (pageOffset != bytesSkipped) {
                long pageLength = pagePath.toFile().length();
                Preconditions.checkArgument((long)pageOffset <= pageLength, "page offset %s exceeded page size %s", pageOffset, pageLength);
                throw new IOException(String.format("Failed to read page %s (%s) from offset %s: %s bytes skipped", pageId, pagePath, pageOffset, bytesSkipped));
            }
            int bytesRead = 0;
            for (int bytesLeft = Math.min((int)target.remaining(), bytesToRead); bytesLeft > 0 && (bytes = target.readFromFile(localFile, bytesLeft)) > 0; bytesRead += bytes, bytesLeft -= bytes) {
            }
            int n = bytesRead;
            return n;
        }
        catch (FileNotFoundException e) {
            throw new PageNotFoundException(pagePath.toString());
        }
    }

    @Override
    public void delete(PageId pageId) throws IOException, PageNotFoundException {
        block16: {
            Path pagePath = this.getPagePath(pageId, false);
            if (!Files.exists(pagePath, new LinkOption[0])) {
                throw new PageNotFoundException(pagePath.toString());
            }
            Files.delete(pagePath);
            Path parent = Preconditions.checkNotNull(pagePath.getParent(), "parent of cache file should not be null");
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(parent);){
                if (!stream.iterator().hasNext()) {
                    Files.delete(parent);
                }
            }
            catch (NoSuchFileException e) {
                if (!LOG.isDebugEnabled()) break block16;
                LOG.debug("Parent path is deleted by other thread, ignore and continue.", (Throwable)e);
            }
        }
    }

    @Override
    public void commit(String fileId, String newFileId) throws IOException {
        Path filePath = this.getFilePath(newFileId);
        Path bucketPath = Preconditions.checkNotNull(filePath.getParent(), "%s does not have a parent path", (Object)filePath);
        if (!Files.exists(bucketPath, new LinkOption[0])) {
            Files.createDirectories(bucketPath, new FileAttribute[0]);
        }
        Files.move(this.getTempFilePath(fileId), filePath, StandardCopyOption.ATOMIC_MOVE);
    }

    @Override
    public void abort(String fileId) throws IOException {
        FileUtils.deleteDirectory(this.getTempFilePath(fileId).toFile());
    }

    private Path getTempFilePath(String fileId) {
        return Paths.get(this.mRoot.toString(), Long.toString(this.mPageSize), TEMP_DIR, fileId);
    }

    private Path getFilePath(String fileId) {
        return Paths.get(this.mRoot.toString(), Long.toString(this.mPageSize), PageStoreDir.getFileBucket(this.mFileBuckets, fileId), fileId);
    }

    @VisibleForTesting
    public Path getPagePath(PageId pageId, boolean isTemporary) {
        Path filePath = isTemporary ? this.getTempFilePath(pageId.getFileId()) : this.getFilePath(pageId.getFileId());
        return filePath.resolve(Long.toString(pageId.getPageIndex()));
    }

    @Override
    public void close() {
    }
}

