// Copyright (c) 2011-present, Facebook, Inc.  All rights reserved.
//  This source code is licensed under both the GPLv2 (found in the
//  COPYING file in the root directory) and Apache 2.0 License
//  (found in the LICENSE.Apache file in the root directory).

package alluxio.shaded.client.org.rocksdb;

import java.util.List;

/**
 * CompactionOptions are used in
 * {@link RocksDB#alluxio.shaded.client.com.actFiles(CompactionOptions, ColumnFamilyHandle, List, int, int, CompactionJobInfo)}
 * calls.
 */
public class CompactionOptions extends RocksObject {

  public CompactionOptions() {
    super(newCompactionOptions());
  }

  /**
   * Get the alluxio.shaded.client.com.action output alluxio.shaded.client.com.ression type.
   *
   * See {@link #setCompression(CompressionType)}.
   *
   * @return the alluxio.shaded.client.com.ression type.
   */
  public CompressionType alluxio.shaded.client.com.ression() {
    return CompressionType.getCompressionType(
        alluxio.shaded.client.com.ression(nativeHandle_));
  }

  /**
   * Set the alluxio.shaded.client.com.action output alluxio.shaded.client.com.ression type.
   *
   * Default: snappy
   *
   * If set to {@link CompressionType#DISABLE_COMPRESSION_OPTION},
   * RocksDB will choose alluxio.shaded.client.com.ression type according to the
   * {@link ColumnFamilyOptions#alluxio.shaded.client.com.ressionType()}, taking into account
   * the output level if {@link ColumnFamilyOptions#alluxio.shaded.client.com.ressionPerLevel()}
   * is specified.
   *
   * @param alluxio.shaded.client.com.ression the alluxio.shaded.client.com.ression type to use for alluxio.shaded.client.com.action output.
   *
   * @return the instance of the current Options.
   */
  public CompactionOptions setCompression(final CompressionType alluxio.shaded.client.com.ression) {
    setCompression(nativeHandle_, alluxio.shaded.client.com.ression.getValue());
    return this;
  }

  /**
   * Get the alluxio.shaded.client.com.action output file size limit.
   *
   * See {@link #setOutputFileSizeLimit(long)}.
   *
   * @return the file size limit.
   */
  public long outputFileSizeLimit() {
    return outputFileSizeLimit(nativeHandle_);
  }

  /**
   * Compaction will create files of size {@link #outputFileSizeLimit()}.
   *
   * Default: 2^64-1, which means that alluxio.shaded.client.com.action will create a single file
   *
   * @param outputFileSizeLimit the size limit
   *
   * @return the instance of the current Options.
   */
  public CompactionOptions setOutputFileSizeLimit(
      final long outputFileSizeLimit) {
    setOutputFileSizeLimit(nativeHandle_, outputFileSizeLimit);
    return this;
  }

  /**
   * Get the maximum number of threads that will concurrently perform a
   * alluxio.shaded.client.com.action job.
   *
   * @return the maximum number of threads.
   */
  public int maxSubcompactions() {
    return maxSubcompactions(nativeHandle_);
  }

  /**
   * This value represents the maximum number of threads that will
   * concurrently perform a alluxio.shaded.client.com.action job by breaking it into multiple,
   * smaller ones that are run simultaneously.
   *
   * Default: 0 (i.e. no subcompactions)
   *
   * If &gt; 0, it will replace the option in
   * {@link DBOptions#maxSubcompactions()} for this alluxio.shaded.client.com.action.
   *
   * @param maxSubcompactions The maximum number of threads that will
   *     concurrently perform a alluxio.shaded.client.com.action job
   *
   * @return the instance of the current Options.
   */
  public CompactionOptions setMaxSubcompactions(final int maxSubcompactions) {
    setMaxSubcompactions(nativeHandle_, maxSubcompactions);
    return this;
  }

  private static native long newCompactionOptions();
  @Override protected final native void disposeInternal(final long handle);

  private static native byte alluxio.shaded.client.com.ression(final long handle);
  private static native void setCompression(final long handle,
      final byte alluxio.shaded.client.com.ressionTypeValue);
  private static native long outputFileSizeLimit(final long handle);
  private static native void setOutputFileSizeLimit(final long handle,
      final long outputFileSizeLimit);
  private static native int maxSubcompactions(final long handle);
  private static native void setMaxSubcompactions(final long handle,
      final int maxSubcompactions);
}
