/*
 * Copyright 2011-2020 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in alluxio.shaded.client.com.liance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.alluxio.shaded.client.com.apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package alluxio.shaded.client.com.amazonaws.regions;


import alluxio.shaded.client.com.amazonaws.SdkClientException;
import alluxio.shaded.client.com.amazonaws.annotation.SdkTestInternalApi;
import alluxio.shaded.client.com.amazonaws.auth.profile.internal.AllProfiles;
import alluxio.shaded.client.com.amazonaws.auth.profile.internal.AwsProfileNameLoader;
import alluxio.shaded.client.com.amazonaws.auth.profile.internal.BasicProfile;
import alluxio.shaded.client.com.amazonaws.auth.profile.internal.BasicProfileConfigLoader;
import alluxio.shaded.client.com.amazonaws.profile.path.AwsProfileFileLocationProvider;
import alluxio.shaded.client.com.amazonaws.util.StringUtils;

import java.alluxio.shaded.client.io.File;

/**
 * Loads region information from the shared AWS config file. Uses the default profile unless
 * otherwise specified.
 */
public class AwsProfileRegionProvider extends AwsRegionProvider {

    private final String profileName;
    private final AwsProfileFileLocationProvider locationProvider;
    private final BasicProfileConfigLoader profileConfigLoader;

    public AwsProfileRegionProvider() {
        this(AwsProfileNameLoader.INSTANCE.loadProfileName());
    }

    public AwsProfileRegionProvider(String profileName) {
        this(profileName, AwsProfileFileLocationProvider.DEFAULT_CONFIG_LOCATION_PROVIDER,
             BasicProfileConfigLoader.INSTANCE);
    }

    @SdkTestInternalApi
    AwsProfileRegionProvider(String profileName, AwsProfileFileLocationProvider locationProvider,
                             BasicProfileConfigLoader configLoader) {
        this.profileName = profileName;
        this.locationProvider = locationProvider;
        this.profileConfigLoader = configLoader;
    }

    @Override
    public String getRegion() throws SdkClientException {
        File configFile = locationProvider.getLocation();
        if (configFile != null && configFile.exists()) {
            BasicProfile profile = loadProfile(configFile);
            if (profile != null && !StringUtils.isNullOrEmpty(profile.getRegion())) {
                return profile.getRegion();
            }
        }
        return null;
    }

    private BasicProfile loadProfile(File configFile) {
        final AllProfiles allProfiles = profileConfigLoader.loadProfiles(configFile);
        return allProfiles.getProfile(profileName);
    }

}
