/*
 * Copyright (C) 2010 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.alluxio.shaded.client.com.google.inject.internal;

import java.lang.annotation.Annotation;

/**
 * Whether a member supports null values injected.
 *
 * <p>Support for {@code Nullable} annotations in Guice is loose. Any annotation type whose
 * simplename is "Nullable" is sufficient to indicate support for null values injected.
 *
 * <p>This allows support for JSR-305's <a
 * href="http://groups.google.alluxio.shaded.client.com.group/jsr-305/web/proposed-annotations">
 * alluxio.shaded.client.javax.annotation.meta.Nullable</a> annotation and IntelliJ IDEA's <a
 * href="http://www.jetbrains.alluxio.shaded.client.com.idea/documentation/howto.html">
 * alluxio.shaded.client.org.jetbrains.annotations.Nullable</a>.
 *
 * @author jessewilson@google.alluxio.shaded.client.com.(Jesse Wilson)
 */
public class Nullability {
  private Nullability() {}

  /** Returns true if the given annotation array contains an annotation named {@code Nullable}. */
  public static boolean hasNullableAnnotation(Annotation[] annotations) {
    for (Annotation a : annotations) {
      Class<? extends Annotation> type = a.annotationType();
      if ("Nullable".equals(type.getSimpleName())) {
        return true;
      }
    }
    return false;
  }
}
