/*
 * Decompiled with CFR 0.152.
 */
package alluxio.util;

import alluxio.shaded.client.com.google.common.annotations.VisibleForTesting;
import alluxio.shaded.client.com.google.common.base.Joiner;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.com.google.common.base.Stopwatch;
import alluxio.shaded.client.com.google.common.collect.Lists;
import alluxio.shaded.client.com.google.common.collect.Sets;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.ObjectName;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

@NotThreadSafe
public class JvmPauseMonitor {
    private static final Log LOG = LogFactory.getLog(JvmPauseMonitor.class);
    private final long mGcSleepIntervalMs;
    private final long mWarnThresholdMs;
    private final long mInfoThresholdMs;
    private final AtomicLong mWarnTimeExceeded = new AtomicLong();
    private final AtomicLong mInfoTimeExceeded = new AtomicLong();
    private final AtomicLong mTotalExtraTimeMs = new AtomicLong();
    private Thread mJvmMonitorThread;

    public JvmPauseMonitor(long gcSleepIntervalMs, long warnThresholdMs, long infoThresholdMs) {
        Preconditions.checkArgument(gcSleepIntervalMs > 0L, "gc sleep interval must be > 0");
        Preconditions.checkArgument(warnThresholdMs > 0L, "warn threshold must be > 0");
        Preconditions.checkArgument(infoThresholdMs > 0L, "info threshold must be > 0");
        Preconditions.checkArgument(warnThresholdMs > infoThresholdMs, "gc warn threshold must be > gc info threshold");
        this.mGcSleepIntervalMs = gcSleepIntervalMs;
        this.mWarnThresholdMs = warnThresholdMs;
        this.mInfoThresholdMs = infoThresholdMs;
    }

    public void start() {
        Preconditions.checkState(this.mJvmMonitorThread == null, "JVM monitor thread already started");
        this.mJvmMonitorThread = new GcMonitor();
        this.mJvmMonitorThread.start();
    }

    public void stop() {
        if (this.mJvmMonitorThread == null) {
            return;
        }
        this.mJvmMonitorThread.interrupt();
        try {
            this.mJvmMonitorThread.join(5000L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.mJvmMonitorThread = null;
    }

    public boolean isStarted() {
        return this.mJvmMonitorThread != null && this.mJvmMonitorThread.isAlive();
    }

    public long getWarnTimeExceeded() {
        return this.mWarnTimeExceeded.get();
    }

    public long getInfoTimeExceeded() {
        return this.mInfoTimeExceeded.get();
    }

    public long getTotalExtraTime() {
        return this.mTotalExtraTimeMs.get();
    }

    private String getMemoryInfo() {
        Runtime runtime = Runtime.getRuntime();
        return "max memory = " + runtime.maxMemory() / 0x100000L + "M total memory = " + runtime.totalMemory() / 0x100000L + "M free memory = " + runtime.freeMemory() / 0x100000L + "M";
    }

    private String formatLogString(long extraSleepTime, Map<String, GarbageCollectorMXBean> gcMXBeanMapBeforeSleep, Map<String, GarbageCollectorMXBean> gcMXBeanMapAfterSleep) {
        ArrayList<String> beanDiffs = Lists.newArrayList();
        Sets.SetView<String> nameSet = Sets.union(gcMXBeanMapBeforeSleep.keySet(), gcMXBeanMapAfterSleep.keySet());
        for (String name : nameSet) {
            GarbageCollectorMXBean oldBean = gcMXBeanMapBeforeSleep.get(name);
            GarbageCollectorMXBean newBean = gcMXBeanMapAfterSleep.get(name);
            if (oldBean == null) {
                beanDiffs.add(String.format("new GCBean created name='%s' count=%d time=%dms", newBean.getName(), newBean.getCollectionCount(), newBean.getCollectionTime()));
                continue;
            }
            if (newBean == null) {
                beanDiffs.add(String.format("old GCBean canceled name= '%s' count=%d time=%dms", oldBean.getName(), oldBean.getCollectionCount(), oldBean.getCollectionTime()));
                continue;
            }
            if (oldBean.getCollectionTime() == newBean.getCollectionTime() && oldBean.getCollectionCount() == newBean.getCollectionCount()) continue;
            beanDiffs.add(String.format("GC name='%s' count=%d time=%dms", newBean.getName(), newBean.getCollectionCount() - oldBean.getCollectionCount(), newBean.getCollectionTime() - oldBean.getCollectionTime()));
        }
        StringBuilder ret = new StringBuilder().append("JVM paused ").append(extraSleepTime).append("ms\n");
        if (beanDiffs.isEmpty()) {
            ret.append("No GCs detected ");
        } else {
            ret.append("GC list:\n" + Joiner.on("\n").join(beanDiffs));
        }
        ret.append("\n").append(this.getMemoryInfo());
        return ret.toString();
    }

    private Map<String, GarbageCollectorMXBean> getGarbageCollectorMXBeans() {
        List<GarbageCollectorMXBean> gcBeanList = ManagementFactory.getGarbageCollectorMXBeans();
        HashMap<String, GarbageCollectorMXBean> gcBeanMap = new HashMap<String, GarbageCollectorMXBean>();
        for (GarbageCollectorMXBean gcBean : gcBeanList) {
            gcBeanMap.put(gcBean.getName(), new GarbageCollectorMXBeanView(gcBean));
        }
        return gcBeanMap;
    }

    @VisibleForTesting
    void sleepMillis(long ms) throws InterruptedException {
        Thread.sleep(ms);
    }

    public static class GarbageCollectorMXBeanView
    implements GarbageCollectorMXBean {
        private final long mCollectionCount;
        private final long mCollectionTime;
        private final String mName;
        private final boolean mValid;
        private final String[] mMemoryPoolNames;
        private final ObjectName mObjectName;

        public GarbageCollectorMXBeanView(GarbageCollectorMXBean gcBean) {
            this.mCollectionCount = gcBean.getCollectionCount();
            this.mCollectionTime = gcBean.getCollectionTime();
            this.mName = gcBean.getName();
            this.mValid = gcBean.isValid();
            this.mMemoryPoolNames = gcBean.getMemoryPoolNames();
            this.mObjectName = gcBean.getObjectName();
        }

        @Override
        public long getCollectionCount() {
            return this.mCollectionCount;
        }

        @Override
        public long getCollectionTime() {
            return this.mCollectionTime;
        }

        @Override
        public String getName() {
            return this.mName;
        }

        @Override
        public boolean isValid() {
            return this.mValid;
        }

        @Override
        public String[] getMemoryPoolNames() {
            return this.mMemoryPoolNames;
        }

        @Override
        public ObjectName getObjectName() {
            return this.mObjectName;
        }
    }

    private class GcMonitor
    extends Thread {
        public GcMonitor() {
            this.setDaemon(true);
        }

        @Override
        public void run() {
            Stopwatch sw = Stopwatch.createUnstarted();
            Map<String, GarbageCollectorMXBean> gcBeanMapBeforeSleep = JvmPauseMonitor.this.getGarbageCollectorMXBeans();
            while (!Thread.currentThread().isInterrupted()) {
                sw.reset().start();
                try {
                    JvmPauseMonitor.this.sleepMillis(JvmPauseMonitor.this.mGcSleepIntervalMs);
                }
                catch (InterruptedException ie) {
                    LOG.warn((Object)"JVM pause monitor interrupted during sleep.");
                    return;
                }
                long extraTime = sw.elapsed(TimeUnit.MILLISECONDS) - JvmPauseMonitor.this.mGcSleepIntervalMs;
                JvmPauseMonitor.this.mTotalExtraTimeMs.addAndGet(extraTime);
                Map<String, GarbageCollectorMXBean> gcBeanMapAfterSleep = JvmPauseMonitor.this.getGarbageCollectorMXBeans();
                if (extraTime > JvmPauseMonitor.this.mWarnThresholdMs) {
                    JvmPauseMonitor.this.mInfoTimeExceeded.incrementAndGet();
                    JvmPauseMonitor.this.mWarnTimeExceeded.incrementAndGet();
                    LOG.warn((Object)JvmPauseMonitor.this.formatLogString(extraTime, gcBeanMapBeforeSleep, gcBeanMapAfterSleep));
                } else if (extraTime > JvmPauseMonitor.this.mInfoThresholdMs) {
                    JvmPauseMonitor.this.mInfoTimeExceeded.incrementAndGet();
                    LOG.info((Object)JvmPauseMonitor.this.formatLogString(extraTime, gcBeanMapBeforeSleep, gcBeanMapAfterSleep));
                }
                gcBeanMapBeforeSleep = gcBeanMapAfterSleep;
            }
        }
    }
}

