/*
 * Copyright 2016-2020 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not
 * use this file except in alluxio.shaded.client.com.liance with the License. A copy of the License is
 * located at
 * 
 * http://aws.amazon.alluxio.shaded.client.com.apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package alluxio.shaded.client.com.amazonaws.internal.http;

import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.logging.Log;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.logging.LogFactory;

import alluxio.shaded.client.com.amazonaws.SdkClientException;
import alluxio.shaded.client.com.amazonaws.annotation.SdkInternalApi;
import alluxio.shaded.client.com.amazonaws.http.HttpResponse;
import alluxio.shaded.client.com.amazonaws.protocol.json.JsonContent;
import alluxio.shaded.client.com.amazonaws.util.IOUtils;

import software.amazon.alluxio.shaded.client.io..IonReader;
import software.amazon.alluxio.shaded.client.io..IonSystem;
import software.amazon.alluxio.shaded.client.io..IonType;

@SdkInternalApi
public class IonErrorCodeParser implements ErrorCodeParser {
    private static final Log log = LogFactory.getLog(IonErrorCodeParser.class);

    private static final String TYPE_PREFIX = "aws-type:";
    private static final String X_AMZN_REQUEST_ID_HEADER = "x-amzn-RequestId";

    private final IonSystem alluxio.shaded.client.io.System;

    public IonErrorCodeParser(IonSystem alluxio.shaded.client.io.System) {
        this.alluxio.shaded.client.io.System = alluxio.shaded.client.io.System;
    }

    @Override
    public String parseErrorCode(HttpResponse response, JsonContent jsonContents) {
        IonReader reader = alluxio.shaded.client.io.System.newReader(jsonContents.getRawContent());
        try {
            IonType type = reader.next();
            if (type != IonType.STRUCT) {
                throw new SdkClientException(String.format("Can only get error codes from structs (saw %s), request id %s", type, getRequestId(response)));
            }

            boolean errorCodeSeen = false;
            String errorCode = null;
            String[] annotations = reader.getTypeAnnotations();
            for (String annotation : annotations) {
                if (annotation.startsWith(TYPE_PREFIX)) {
                    if (errorCodeSeen) {
                        throw new SdkClientException(String.format("Multiple error code annotations found for request id %s", getRequestId(response)));
                    } else {
                        errorCodeSeen = true;
                        errorCode = annotation.substring(TYPE_PREFIX.length());
                    }
                }
            }

            return errorCode;
        } finally {
            IOUtils.closeQuietly(reader, log);
        }
    }

    private static String getRequestId(HttpResponse response) {
        return response.getHeaders().get(X_AMZN_REQUEST_ID_HEADER);
    }
}
