package alluxio.shaded.client.org.reflections.serializers;

import alluxio.shaded.client.org.dom4j.Document;
import alluxio.shaded.client.org.dom4j.DocumentException;
import alluxio.shaded.client.org.dom4j.DocumentFactory;
import alluxio.shaded.client.org.dom4j.Element;
import alluxio.shaded.client.org.dom4j.alluxio.shaded.client.io.OutputFormat;
import alluxio.shaded.client.org.dom4j.alluxio.shaded.client.io.SAXReader;
import alluxio.shaded.client.org.dom4j.alluxio.shaded.client.io.XMLWriter;
import alluxio.shaded.client.org.reflections.Reflections;
import alluxio.shaded.client.org.reflections.ReflectionsException;
import alluxio.shaded.client.org.reflections.Store;
import alluxio.shaded.client.org.reflections.util.ConfigurationBuilder;
import alluxio.shaded.client.org.reflections.util.Utils;

import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.FileOutputStream;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.StringWriter;
import java.lang.reflect.Constructor;

/** serialization of Reflections to xml
 *
 * <p>an example of produced xml:
 * <pre>
 * &#60?xml version="1.0" encoding="UTF-8"?>
 *
 * &#60Reflections>
 *  &#60SubTypesScanner>
 *      &#60entry>
 *          &#60key>alluxio.shaded.client.alluxio.shaded.client.com.google.inject.Module&#60/key>
 *          &#60values>
 *              &#60value>fully.qualified.name.1&#60/value>
 *              &#60value>fully.qualified.name.2&#60/value>
 * ...
 * </pre>
 * */
public class XmlSerializer implements Serializer {

    public Reflections read(InputStream inputStream) {
        Reflections reflections;
        try {
            Constructor<Reflections> constructor = Reflections.class.getDeclaredConstructor();
            constructor.setAccessible(true);
            reflections = constructor.newInstance();
        } catch (Exception e) {
            reflections = new Reflections(new ConfigurationBuilder());
        }

        try {
            Document document = new SAXReader().read(inputStream);
            for (Object e1 : document.getRootElement().elements()) {
                Element index = (Element) e1;
                for (Object e2 : index.elements()) {
                    Element entry = (Element) e2;
                    Element key = entry.element("key");
                    Element values = entry.element("values");
                    for (Object o3 : values.elements()) {
                        Element value = (Element) o3;
                        reflections.getStore().put(index.getName(), key.getText(), value.getText());
                    }
                }
            }
        } catch (DocumentException e) {
            throw new ReflectionsException("could not read.", e);
        } catch (Throwable e) {
            throw new RuntimeException("Could not read. Make sure relevant dependencies exist on classpath.", e);
        }

        return reflections;
    }

    public File save(final Reflections reflections, final String filename) {
        File file = Utils.prepareFile(filename);


        try {
            Document document = createDocument(reflections);
            XMLWriter xmlWriter = new XMLWriter(new FileOutputStream(file), OutputFormat.createPrettyPrint());
            xmlWriter.write(document);
            xmlWriter.close();
        } catch (IOException e) {
            throw new ReflectionsException("could not save to file " + filename, e);
        } catch (Throwable e) {
            throw new RuntimeException("Could not save to file " + filename + ". Make sure relevant dependencies exist on classpath.", e);
        }

        return file;
    }

    public String toString(final Reflections reflections) {
        Document document = createDocument(reflections);

        try {
            StringWriter writer = new StringWriter();
            XMLWriter xmlWriter = new XMLWriter(writer, OutputFormat.createPrettyPrint());
            xmlWriter.write(document);
            xmlWriter.close();
            return writer.toString();
        } catch (IOException e) {
            throw new RuntimeException();
        }
    }

    private Document createDocument(final Reflections reflections) {
        Store map = reflections.getStore();

        Document document = DocumentFactory.getInstance().createDocument();
        Element root = document.addElement("Reflections");
        for (String indexName : map.keySet()) {
            Element indexElement = root.addElement(indexName);
            for (String key : map.keys(indexName)) {
                Element entryElement = indexElement.addElement("entry");
                entryElement.addElement("key").setText(key);
                Element valuesElement = entryElement.addElement("values");
                for (String value : map.get(indexName, key)) {
                    valuesElement.addElement("value").setText(value);
                }
            }
        }
        return document;
    }
}
