/*
 * Javassist, a Java-bytecode translator toolkit.
 * Copyright (C) 1999- Shigeru Chiba. All Rights Reserved.
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance with
 * the License.  Alternatively, the contents of this file may be used under
 * the terms of the GNU Lesser General Public License Version 2.1 or later,
 * or the Apache License Version 2.0.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 */

package alluxio.shaded.client.javassist.bytecode;

import java.alluxio.shaded.client.io.DataInputStream;
import java.alluxio.shaded.client.io.IOException;
import java.util.Map;

/**
 * <code>NestHost_attribute</code>.
 * It was introduced by JEP-181.  See JVMS 4.7.28 for the specification.
 *
 * @since 3.24
 */
public class NestHostAttribute extends AttributeInfo {
    /**
     * The name of this attribute <code>"NestHost"</code>.
     */
    public static final String tag = "NestHost";

    NestHostAttribute(ConstPool cp, int n, DataInputStream in) throws IOException {
        super(cp, n, in);
    }

    private NestHostAttribute(ConstPool cp, int hostIndex) {
        super(cp, tag, new byte[2]);
        ByteArray.write16bit(hostIndex, get(), 0);
    }

    /**
     * Makes a copy.  Class names are replaced according to the
     * given <code>Map</code> object.
     *
     * @param newCp     the constant pool table used by the new copy.
     * @param classnames        pairs of replaced and substituted
     *                          class names.
     */
    @Override
    public AttributeInfo copy(ConstPool newCp, Map<String, String> classnames) {
        int hostIndex = ByteArray.readU16bit(get(), 0);
        int newHostIndex = getConstPool().copy(hostIndex, newCp, classnames);
        return new NestHostAttribute(newCp, newHostIndex);
    }

    /**
     * Returns <code>host_class_index</code>.  The constant pool entry
     * at this entry is a <code>CONSTANT_Class_info</code> structure.
     * @return the value of <code>host_class_index</code>.
     */
    public int hostClassIndex() {
        return ByteArray.readU16bit(info, 0);
    }
}
