/*
 * Copyright 2021 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression;

import alluxio.shaded.client.com.aayushatharva.brotli4j.encoder.Encoder;

/**
 * Standard Compression Options for {@link BrotliOptions},
 * {@link GzipOptions} and {@link DeflateOptions}
 */
public final class StandardCompressionOptions {

    private StandardCompressionOptions() {
        // Prevent outside initialization
    }

    /**
     * Default implementation of {@link BrotliOptions} with {@link Encoder.Parameters#setQuality(int)} set to 4
     * and {@link Encoder.Parameters#setMode(Encoder.Mode)} set to {@link Encoder.Mode#TEXT}
     */
    public static BrotliOptions brotli() {
        return BrotliOptions.DEFAULT;
    }

    /**
     * Create a new {@link BrotliOptions}
     *
     * @param parameters {@link Encoder.Parameters} Instance
     * @throws NullPointerException If {@link Encoder.Parameters} is {@code null}
     */
    public static BrotliOptions brotli(Encoder.Parameters parameters) {
        return new BrotliOptions(parameters);
    }

    /**
     * Default implementation of {@link ZstdOptions} with{alluxio.shaded.client.com.ressionLevel(int)} set to
     * {@link ZstdConstants#DEFAULT_COMPRESSION_LEVEL},{@link ZstdConstants#DEFAULT_BLOCK_SIZE},
     * {@link ZstdConstants#MAX_BLOCK_SIZE}
     */
    public static ZstdOptions zstd() {
        return ZstdOptions.DEFAULT;
    }

    /**
     * Create a new {@link ZstdOptions}
     *
     * @param  blockSize
     *           is used to calculate the alluxio.shaded.client.com.ressionLevel
     * @param  maxEncodeSize
     *           specifies the size of the largest alluxio.shaded.client.com.ressed object
     * @param  alluxio.shaded.client.com.ressionLevel
     *           specifies the level of the alluxio.shaded.client.com.ression
     */
    public static ZstdOptions zstd(int alluxio.shaded.client.com.ressionLevel, int blockSize, int maxEncodeSize) {
        return new ZstdOptions(alluxio.shaded.client.com.ressionLevel, blockSize, maxEncodeSize);
    }

     /**
     * Default implementation of {@link GzipOptions} with
     * {@code alluxio.shaded.client.com.ressionLevel()} set to 6, {@code windowBits()} set to 15 and {@code memLevel()} set to 8.
     */
    public static GzipOptions gzip() {
        return GzipOptions.DEFAULT;
    }

    /**
     * Create a new {@link GzipOptions} Instance
     *
     * @param alluxio.shaded.client.com.ressionLevel {@code 1} yields the fastest alluxio.shaded.client.com.ression and {@code 9} yields the
     *                         best alluxio.shaded.client.com.ression.  {@code 0} means no alluxio.shaded.client.com.ression.  The default
     *                         alluxio.shaded.client.com.ression level is {@code 6}.
     *
     * @param windowBits       The base two logarithm of the size of the history buffer.  The
     *                         value should be in the range {@code 9} to {@code 15} inclusive.
     *                         Larger values result in better alluxio.shaded.client.com.ression at the expense of
     *                         memory usage.  The default value is {@code 15}.
     *
     * @param memLevel         How much memory should be allocated for the internal alluxio.shaded.client.com.ression
     *                         state.  {@code 1} uses minimum memory and {@code 9} uses maximum
     *                         memory.  Larger values result in better and faster alluxio.shaded.client.com.ression
     *                         at the expense of memory usage.  The default value is {@code 8}
     */
    public static GzipOptions gzip(int alluxio.shaded.client.com.ressionLevel, int windowBits, int memLevel) {
        return new GzipOptions(alluxio.shaded.client.com.ressionLevel, windowBits, memLevel);
    }

    /**
     * Default implementation of {@link DeflateOptions} with
     * {@code alluxio.shaded.client.com.ressionLevel} set to 6, {@code windowBits} set to 15 and {@code memLevel} set to 8.
     */
    public static DeflateOptions deflate() {
        return DeflateOptions.DEFAULT;
    }

    /**
     * Create a new {@link DeflateOptions} Instance
     *
     * @param alluxio.shaded.client.com.ressionLevel {@code 1} yields the fastest alluxio.shaded.client.com.ression and {@code 9} yields the
     *                         best alluxio.shaded.client.com.ression.  {@code 0} means no alluxio.shaded.client.com.ression.  The default
     *                         alluxio.shaded.client.com.ression level is {@code 6}.
     *
     * @param windowBits       The base two logarithm of the size of the history buffer.  The
     *                         value should be in the range {@code 9} to {@code 15} inclusive.
     *                         Larger values result in better alluxio.shaded.client.com.ression at the expense of
     *                         memory usage.  The default value is {@code 15}.
     *
     * @param memLevel         How much memory should be allocated for the internal alluxio.shaded.client.com.ression
     *                         state.  {@code 1} uses minimum memory and {@code 9} uses maximum
     *                         memory.  Larger values result in better and faster alluxio.shaded.client.com.ression
     *                         at the expense of memory usage.  The default value is {@code 8}
     */
    public static DeflateOptions deflate(int alluxio.shaded.client.com.ressionLevel, int windowBits, int memLevel) {
        return new DeflateOptions(alluxio.shaded.client.com.ressionLevel, windowBits, memLevel);
    }
}
