/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package alluxio.shaded.client.io.vertx.core.http.impl;

import alluxio.shaded.client.io.netty.channel.ChannelPipeline;
import alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZlibCodecFactory;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtensionHandler;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.WebSocketServerExtensionHandshaker;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression.DeflateFrameServerExtensionHandshaker;
import alluxio.shaded.client.io.netty.handler.codec.http.websocketx.extensions.alluxio.shaded.client.com.ression.PerMessageDeflateServerExtensionHandshaker;
import alluxio.shaded.client.io.vertx.core.Handler;
import alluxio.shaded.client.io.vertx.core.http.HttpConnection;
import alluxio.shaded.client.io.vertx.core.http.HttpServerRequest;
import alluxio.shaded.client.io.vertx.core.http.ServerWebSocket;
import alluxio.shaded.client.io.vertx.core.impl.ContextInternal;

import java.util.ArrayList;

/**
 * HTTP server connection handler.
 *
 * @author <a href="mailto:julien@julienviet.alluxio.shaded.client.com.>Julien Viet</a>
 */
public class HttpServerConnectionHandler implements Handler<HttpServerConnection> {

  final HttpServerImpl server;
  final Handler<HttpServerRequest> requestHandler;
  final Handler<HttpServerRequest> invalidRequestHandler;
  final Handler<ServerWebSocket> wsHandler;
  final Handler<HttpConnection> connectionHandler;
  final Handler<Throwable> exceptionHandler;

  public HttpServerConnectionHandler(
    HttpServerImpl server,
    Handler<HttpServerRequest> requestHandler,
    Handler<HttpServerRequest> invalidRequestHandler,
    Handler<ServerWebSocket> wsHandler,
    Handler<HttpConnection> connectionHandler,
    Handler<Throwable> exceptionHandler) {
    this.server = server;
    this.requestHandler = requestHandler;
    this.invalidRequestHandler = invalidRequestHandler == null ? HttpServerRequest.DEFAULT_INVALID_REQUEST_HANDLER : invalidRequestHandler;
    this.wsHandler = wsHandler;
    this.connectionHandler = connectionHandler;
    this.exceptionHandler = exceptionHandler;
  }

  @Override
  public void handle(HttpServerConnection conn) {
    Handler<HttpServerRequest> requestHandler = this.requestHandler;
    if (HttpServerImpl.DISABLE_WEBSOCKETS) {
      // As a performance optimisation you can set a system property to disable WebSockets altogether which avoids
      // some casting and a header check
    } else {
      if (conn instanceof Http1xServerConnection) {
        requestHandler =  new Http1xServerRequestHandler(this);
        Http1xServerConnection c = (Http1xServerConnection) conn;
        initializeWebSocketExtensions(c.channelHandlerContext().pipeline());
      }
    }
    conn.exceptionHandler(exceptionHandler);
    conn.handler(requestHandler);
    conn.invalidRequestHandler(invalidRequestHandler);
    if (connectionHandler != null) {
      // We hand roll event-loop execution in case of a worker context
      ContextInternal ctx = conn.getContext();
      ContextInternal prev = ctx.beginDispatch();
      try {
        connectionHandler.handle(conn);
      } catch (Exception e) {
        ctx.reportException(e);
      } finally {
        ctx.endDispatch(prev);
      }
    }
  }

  private void initializeWebSocketExtensions(ChannelPipeline pipeline) {
    ArrayList<WebSocketServerExtensionHandshaker> extensionHandshakers = new ArrayList<>();
    if (server.options.getPerFrameWebSocketCompressionSupported()) {
      extensionHandshakers.add(new DeflateFrameServerExtensionHandshaker(server.options.getWebSocketCompressionLevel()));
    }
    if (server.options.getPerMessageWebSocketCompressionSupported()) {
      extensionHandshakers.add(new PerMessageDeflateServerExtensionHandshaker(server.options.getWebSocketCompressionLevel(),
        ZlibCodecFactory.isSupportingWindowSizeAndMemLevel(), PerMessageDeflateServerExtensionHandshaker.MAX_WINDOW_SIZE,
        server.options.getWebSocketAllowServerNoContext(), server.options.getWebSocketPreferredClientNoContext()));
    }
    if (!extensionHandshakers.isEmpty()) {
      WebSocketServerExtensionHandler extensionHandler = new WebSocketServerExtensionHandler(
        extensionHandshakers.toArray(new WebSocketServerExtensionHandshaker[0]));
      pipeline.addBefore("handler", "webSocketExtensionHandler", extensionHandler);
    }
  }
}
