/*
 * Copyright 2010-2020 Amazon.alluxio.shaded.client.com. Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in alluxio.shaded.client.com.liance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.alluxio.shaded.client.com.apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package alluxio.shaded.client.com.amazonaws.monitoring;

import alluxio.shaded.client.com.amazonaws.SdkClientException;
import alluxio.shaded.client.com.amazonaws.annotation.ThreadSafe;

import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.AWS_CSM_CLIENT_ID_SYSTEM_PROPERTY;
import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.AWS_CSM_ENABLED_SYSTEM_PROPERTY;
import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.AWS_CSM_HOST_SYSTEM_PROPERTY;
import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.AWS_CSM_PORT_SYSTEM_PROPERTY;
import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.DEFAULT_AWS_CSM_PORT;
import static alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration.DEFAULT_AWS_CSM_HOST;

/**
 * Configuration provider that sources the client side monitoring
 * configuration parameters from system properties.
 *
 * @see alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration#AWS_CSM_CLIENT_ID_SYSTEM_PROPERTY
 * @see alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration#AWS_CSM_ENABLED_SYSTEM_PROPERTY
 * @see alluxio.shaded.client.com.amazonaws.SDKGlobalConfiguration#AWS_CSM_PORT_SYSTEM_PROPERTY
 */
@ThreadSafe
public final class SystemPropertyCsmConfigurationProvider implements CsmConfigurationProvider {
    @Override
    public CsmConfiguration getConfiguration() throws SdkClientException {
        String enabled = System.getProperty(AWS_CSM_ENABLED_SYSTEM_PROPERTY);

        if (enabled == null) {
            throw new SdkClientException("Unable to load Client Side Monitoring configurations from"
                    + " system properties variables!");
        }

        String host = System.getProperty(AWS_CSM_HOST_SYSTEM_PROPERTY, DEFAULT_AWS_CSM_HOST);
        String port = System.getProperty(AWS_CSM_PORT_SYSTEM_PROPERTY);
        String clientId = System.getProperty(AWS_CSM_CLIENT_ID_SYSTEM_PROPERTY, "");

        try {
            int portNumber = port == null ? DEFAULT_AWS_CSM_PORT : Integer.parseInt(port);
            return CsmConfiguration.builder()
                    .withEnabled(Boolean.parseBoolean(enabled))
                    .withHost(host)
                    .withPort(portNumber)
                    .withClientId(clientId)
                    .build();
        } catch (Exception e) {
            throw new SdkClientException("Unable to load Client Side Monitoring configurations from"
                    + " system properties variables!", e);
        }
    }
}
